/*
* TurtleKit - A 'reactive simulation platform' using MadKit Kernel
* Copyright (C) 2000-2007 Fabien Michel, Gregory Beurier
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/


package turtlekit2.kernel;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.Vector;
import javax.xml.parsers.DocumentBuilderFactory;
import madkit.boot.Madkit;
import turtlekit2.genetic.*;
import madkit.kernel.AbstractAgent;
import madkit.kernel.Agent;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * <p>Title: TurtleKit Launcher </p>
 * This agent sets up, launches (after xml loading and parsing) and manages Turtle based simulations. All simulation parameters
 * are loaded from an xml file (Open Simulation).
 * 
 * <p></p>
 * <p>XML Attributes : for the simulations parameters. (Simulation Node)</p>
 * <p>Name: the name of the simulation. default is Turtlekit Simulation</p>
 * <p>Refresh: the step of reseting simulations (except if a manager resets itself the simulation). default is 0 (no refresh)</p>
 * <p>Width: the width of the environment. default is 100.</p>
 * <p>Height: the height of the environment. default is 100.</p>
 * <p>CellSize: the graphical scale of the environment. default is 3 (for a cell).</p>
 * <p>TorusMode: the type of environment. "on" is a torus world. "off" a closed one. default is on.</p>
 * <p>CyclePause: the pause beetween two cycles/steps of simulation. default is 1.</p>
 * <p>--------------------------------------------------------------------------------------------------------------</p>
 * 
 * <p>XML Attributes : for the agents initialisation. (Agents and Agent Nodes)</p>
 * <p>AgentClass: the class of the agent. default is EmitMovingTurtle (in package tools)</p>
 * <p>NbAgents: the number of agents. default is 1.
 * <p>Node "Parameters" is dedicated to specific agent parameters.
 * <p>x: the x coordinate of the agent. default is random.
 * <p>y: the y coordinate of the agent. default is random.
 * <p>--------------------------------------------------------------------------------------------------------------</p>
 * 
 * <p>XML Attributes : for the Viewers initialisation. (Viewers and Viewer Nodes)</p>
 * <p>ViewerClass: the class of the Viewer. default is Viewer (in package kernel)</p>
 * <p>--------------------------------------------------------------------------------------------------------------</p>
 * 
 * <p>XML Attributes : for the Observers initialisation. (Observers and Observer Nodes)</p>
 * <p>ObserverClass: the class of the Observer. default is none.</p>
 * <p>--------------------------------------------------------------------------------------------------------------</p>
 * 
 * <p>XML Attributes : for the Flavors/Pheromones initialisation. (Flavors and Observer/RandomFlavors Nodes)</p>
 * <p>For Standard Flavors (Flavor Node)
 * <p>Name: the name of the flavor . default is "Flavor0","Flavor1", etc...</p>
 * <p>Diffusion: the diffusion coefficient. default is 0.</p>
 * <p>Evaporation: the evaporation rate. default is 0.</p>
 * <p>Quantity: the initial quantity of flavor in this environment. default is 0.</p>
 * <p>Volatile: "on", "off". A volatile flavor is removed from the environment at each generation. </p>
 * 
 * <p>For Random Flavors (RandomFlavors Node)
 * <p>Name: the name of the flavor . default is "RandomFlavor0","RandomFlavor1", etc...</p>
 * <p>Diffusion: the diffusion coefficient. default is random (0-1).</p>
 * <p>Evaporation: the evaporation rate. default is  random (0-1).</p>
 * <p>Quantity: the initial quantity of flavor in this environment. default is  random (0-10000).</p>
 * <p>Number: the number of ramdomly generated flavors. default is 1.</p>
 * <p>Volatile: "on", "off". A volatile flavor is removed from the environment at each generation. </p>
 * <p>--------------------------------------------------------------------------------------------------------------</p>
 * 
 * 
 * DEPRECATED:
 * You have to defined a subclass of this one to custom and launch a simulation. Yo have
 * at least to override the addSimulationAgents method to create the agents you
 * want to launch in your simulation: turtles, viewers and observers. To declare
 * patch variables (PatchVariable objects) you have to override the
 * initializePatchVariables method.
 *
 * @author Fabien MICHEL, Gregory BEURIER
 */

public class Launcher extends Agent {
    //flags
    boolean standAloneMode = false;
    boolean geneticMode = false;
    
    LauncherGui onScreen;
    TurtleEnvironment playGround = null;
    TurtleScheduler sch = null;
    MetaManager metamanager = null;
    
    //Default parameters
    String simulationName = "TurtleKit Simulation";
    int cellSize = 2, cyclePause = 0, cycleDisplayEvery = 1000,envWidth = 100, envHeight = 100, defaultRefresh=0;
    
    Vector<PatchVariable> flavors = null;
    
    //new
    ArrayList<PatchVariable> volatileVariables= new ArrayList<PatchVariable>();
    ArrayList<Turtle> initialPopulation=new ArrayList<Turtle>();
    ArrayList<AbstractAgent> launchedObservers=new ArrayList<AbstractAgent>();
    NodeList simulationsNodes;
    NodeList flavorsList, viewersList, observersList,agentsList, poolsList, managersList;
    Element simulationNode;
    public String simulationFilePath;
    
    
    Class<? extends Viewer> preferedViewer;
    
    final public static String COMMUNITY="Tkit";
    
    boolean run = false, start = false, wrap = true, diffusion = false, pythonOn = false;
    
    
    /**
     * The constructor is where you have to change the default values of the
     * simulation parameters using the corresponding accessors. Default Values
     * are: setSimulationName("? NAME ?"); //The simulation name corresponds to
     * the Madkit group that will be created for the simulation. setWidth(100);
     * setHeight(100); setCellSize(4); //onscreen size for patches and trutles
     * setWrapModeOn(false); setCyclePause(10); //cycle pause represents the
     * pause time between two simulation steps. This default quick pause is
     * supposed to avoid that the simulation takes all ressources.
     */
    public Launcher() {
        simulationFilePath="simulation.xml";
        standAloneMode = true;
                /*
                 * try{ Class python =
                 * Class.forName("org.python.util.PythonInterpreter"); interp =
                 * (org.python.util.PythonInterpreter) python.newInstance(); pythonOn =
                 * true; } catch(Exception e) { pythonOn = false;}
                 */
        
    }
    
    public Launcher(String path) {
        simulationFilePath=path;
        standAloneMode = false;
                /*
                 * try{ Class python =
                 * Class.forName("org.python.util.PythonInterpreter"); interp =
                 * (org.python.util.PythonInterpreter) python.newInstance(); pythonOn =
                 * true; } catch(Exception e) { pythonOn = false;}
                 */
        
    }
    
    public Launcher(Node rootNode){
        standAloneMode = false;
        simulationNode = (Element)rootNode;
    }
    
    
    /**  Xml file loading and parsing initialisation */
    public void loadConfigFile(){
        if(simulationFilePath!=null)
            try {
                /****config file reading****/
                FileInputStream configFile = new FileInputStream(new File(simulationFilePath));
                DocumentBuilderFactory factory  = DocumentBuilderFactory.newInstance();
                Document config =  factory.newDocumentBuilder().parse(configFile);
                simulationNode = config.getDocumentElement();
               // System.out.println(simulationNode);
                
            }catch(IOException e){
                System.err.println("File read error with !\n"+simulationFilePath);
            }catch(SAXException e){
                System.err.println("Load file: Parsing error of the file !\n"+simulationFilePath);
            }catch(Exception e){
                System.err.println("Load file error !\n"+simulationFilePath);
            }
    }
    
    public void setSimulationNode(Element rootNode){
        simulationNode = rootNode;
    }
    
    /** xml Tree nodes init */
    private void initializeNodes(){
        try{
            flavorsList= simulationNode.getElementsByTagName("Flavors");
            viewersList = simulationNode.getElementsByTagName("Viewers");
            observersList = simulationNode.getElementsByTagName("Observers");
            agentsList = simulationNode.getElementsByTagName("Agents"); 
            managersList = simulationNode.getElementsByTagName("Managers");
            NodeList poolsGroup = simulationNode.getElementsByTagName("Pools");
            if(poolsGroup.getLength() > 0){ 
            	geneticMode = true;
            	Node poolList = poolsGroup.item(0);
            	poolsList = ((Element)poolList).getElementsByTagName("Pool");
            }
        }catch(Exception e){System.err.println("Nodes initialisation error" + e);}
    }
    
    
    /** When the simulation parameters are loaded, attributes are given to the environment */
    protected void initializeSimulationParameters() {
        try{
            int wid, hei, cellS, cycleP;
            boolean wrapM;
            if(simulationNode.hasAttribute("Name")) simulationName = simulationNode.getAttribute("Name");
            if(simulationNode.hasAttribute("Width")) wid = Integer.parseInt(simulationNode.getAttribute("Width"));
            else{
                println("width default: 100");
                wid=100;
            }
            if(simulationNode.hasAttribute("Height")) hei = Integer.parseInt(simulationNode.getAttribute("Height"));
            else{
                println("height default: 100");
                hei=100;
            }
            if(simulationNode.hasAttribute("CellSize"))  cellS = Integer.parseInt(simulationNode.getAttribute("CellSize"));
            else {
                println("cellSize default: 3");
                cellS=3;
            }
            if(simulationNode.hasAttribute("TorusMode")) {
                if(simulationNode.getAttribute("TorusMode").equals("on")) wrapM = true;
                else  wrapM = false;
            } else  {
                println("wrapMode default: true");
                wrapM = true;
            }  
            if(simulationNode.hasAttribute("Refresh")) defaultRefresh = Integer.parseInt(simulationNode.getAttribute("Refresh"));
            else{
                println("defaultRefresh default: 0");
                defaultRefresh=0;
            }
            if(simulationNode.hasAttribute("CyclePause"))  cycleP = Integer.parseInt(simulationNode.getAttribute("CyclePause"));
            else {
                println("cyclePause default: 1");
                cycleP=1;
            }
            setWidth(wid);
            setHeight(hei);
            setCellSize(cellS);
            setCyclePause(cycleP);
            setWrapModeOn(wrapM);
        } catch(Exception e){System.err.println("Simulation init: Parsing Error" + e);}
    }
    
    
    /** agent parameters parsing and initialisation */
    public void thisCreateInitPopulation(){
        for(int i=0;i<agentsList.getLength();i++){
            Node agentList = agentsList.item(i);
            NodeList agentsNodes;
            agentsNodes = ((Element)agentList).getElementsByTagName("Agent");
            createInitialPopulation(agentsNodes, i);
        }
    }
    
    
    /** Agents/Turtles population creation based on xml parameters */
    protected void createInitialPopulation(NodeList agentsNodes, int fatherNodeNumber){
        try {
        	System.out.println("AGENT NODE " + agentsNodes.getLength());
        	if(agentsNodes.getLength()>0){
                for(int i=0; i<agentsNodes.getLength(); i++){
                    Element agentDescription = (Element) agentsNodes.item(i);
                    String type = "";
                    if(agentDescription.hasAttribute("AgentClass")) type = agentDescription.getAttribute("AgentClass");
                    else type = "turtlekit2.tools.RandomMovingTurtle";
                    int nbAgents = 1;
                    if(agentDescription.hasAttribute("NbAgents")) nbAgents = Integer.parseInt(agentDescription.getAttribute("NbAgents"));
                    NodeList parametersList = agentDescription.getElementsByTagName("Parameters");
                    Element parameters = (Element)parametersList.item(0);
                    NamedNodeMap attributeList = parameters.getAttributes();
                    int listLength = attributeList.getLength();
                    XMLAttributes agentAttribute = new XMLAttributes();
                    for(int k=0; k<listLength;k++) {
                        agentAttribute.put(attributeList.item(k).getNodeName(),parameters.getAttribute(attributeList.item(k).getNodeName()));
                    }
                    String nodeIdentifier = new Integer(fatherNodeNumber).toString()+new Integer(i).toString();
                    agentAttribute.put("Node",nodeIdentifier);
                    agentAttribute.put("GroupNode",new Integer(fatherNodeNumber).toString());
                    agentAttribute.put("AgentClass",type);
                    if(agentDescription.hasAttribute("x")) agentAttribute.put("x",agentDescription.getAttribute("x"));
                    if(agentDescription.hasAttribute("y")) agentAttribute.put("y",agentDescription.getAttribute("y"));
                    newAgents(agentAttribute, nbAgents);
                }
            }else{
            	String type = "turtlekit2.tools.RandomMovingTurtle";
            	int nbAgents = 10;
            	XMLAttributes agentAttribute = new XMLAttributes();
            	agentAttribute.put("Node","0");
                agentAttribute.put("GroupNode","0");
                agentAttribute.put("AgentClass",type);
                newAgents(agentAttribute, nbAgents);
            }
        } catch(Exception e){
            System.err.println("Agent Initialisation problem !\n" + e);
        }
    }
    
    /** Creation of each agent type */
    @SuppressWarnings("unchecked")
	public void newAgents(XMLAttributes agentAttribute, int nbAgents){
        String agentType = agentAttribute.getString("AgentClass");
        try{
            Class<? extends Turtle> agentClass = (Class<? extends Turtle>) Madkit.getClassLoader().loadClass(agentType);
            for(int j=0; j<nbAgents; j++) {
                Turtle newAgent = (Turtle)(agentClass.newInstance());
                newAgent.setAttributes(agentAttribute);
                if(newAgent.getAttributes().containsKey("x") && newAgent.getAttributes().containsKey("y")) addTurtle(newAgent,newAgent.getAttributes().getInt("x"),newAgent.getAttributes().getInt("y"));
                else addTurtle(newAgent);
                initialPopulation.add(newAgent);
            }
        }catch(Exception e){
            System.err.println("Agent Initialisation problem: "+ agentAttribute + "\n" + e);
        }
    }
    
    /** Viewers parameters Parsing */
    public void thisInitViewers(){
        for(int i=0;i<viewersList.getLength();i++){
            Node viewerList = viewersList.item(i);
            NodeList viewersNodes;
            viewersNodes = ((Element)viewerList).getElementsByTagName("Viewer");
            initializeViewers(viewersNodes);
        }
    }
    
    /** Agents/Viewers population creation based on xml parameters */
    public void initializeViewers(NodeList viewersNodes){
        try {
            if(viewersNodes.getLength()>0){
                for(int i=0; i<viewersNodes.getLength(); i++){
                    Element viewerDescription = (Element) viewersNodes.item(i);
                    NamedNodeMap attributeList = viewerDescription.getAttributes();
                    int listLength = attributeList.getLength();
                    XMLAttributes viewerTable = new XMLAttributes();
                    for(int k=0; k<listLength;k++) {
                        viewerTable.put(attributeList.item(k).getNodeName(),viewerDescription.getAttribute(attributeList.item(k).getNodeName()));
                    }
                    newViewer(viewerTable);
                }
            }else{
            	XMLAttributes viewerTable = new XMLAttributes();
            	newViewer(viewerTable);
            }
        }catch(Exception e){
            System.err.println("Viewers Groups Initialisation problem !\n" + e);
        }
    }
    
    /** Creation of each Viewer type */
    public void newViewer(XMLAttributes viewerParameters){
        String viewerType = "";
        if(viewerParameters.containsKey("ViewerClass")) viewerType = viewerParameters.getString("ViewerClass");
        else viewerType = "turtlekit2.kernel.Viewer";
        try{
            Class viewerClass = Madkit.getClassLoader().loadClass(viewerType);
            Viewer newViewer = (Viewer)(viewerClass.newInstance());
            newViewer.setAttrib(viewerParameters);
            //newViewer.init();
            addViewer(newViewer);
        }catch(Exception e){
            System.err.println("Viewer Initialisation problem: "+ viewerType + "\n" + e);
        }
    }
    
    
    /** Observers parameters Parsing */ 
    public void thisInitObservers(){
        for(int i=0;i<observersList.getLength();i++){
            Node observerList = observersList.item(i);
            NodeList observersNodes;
            observersNodes = ((Element)observerList).getElementsByTagName("Observer");
            initializeObservers(observersNodes);
        }
    }
    
    /** Agents/Observers population creation based on xml parameters */
    public void initializeObservers(NodeList observersNodes){
        try {
            if(observersNodes.getLength()>0){
                for(int i=0; i<observersNodes.getLength(); i++){
                    /*****init patches****/
                    Element observerDescription = (Element) observersNodes.item(i);
                    NamedNodeMap attributeList = observerDescription.getAttributes();
                    int listLength = attributeList.getLength();
                    XMLAttributes observerTable = new XMLAttributes();
                    for(int k=0; k<listLength;k++) {
                        observerTable.put(attributeList.item(k).getNodeName(),observerDescription.getAttribute(attributeList.item(k).getNodeName()));
                    }
                    newObserver(observerTable);
                }
            }
        }catch(Exception e){
            System.err.println("Observer Instanciation problem !\n" + e);
        }
    }
    
    /** Creation of each Observer type */
    public void newObserver(XMLAttributes observerParameters){
        String observerType = observerParameters.getString("ObserverClass");
        try{
            Class observerClass = Madkit.getClassLoader().loadClass(observerType);
            Observer newObserver = (Observer)(observerClass.newInstance());
            newObserver.setAttrib(observerParameters);
            //newObserver.init();
            addObserver(newObserver,false);
        }catch(Exception e){
            System.err.println("Observer Instanciation problem: "+ observerType + "\n" + e);
        }
        System.out.println("Observer init:" + observerParameters);
    }
    
    /** display method */
    public void launch(Node n){
        System.out.println(n.getNodeName());
    }

    /** flavors parameters parsing and initialisation */
    public void thisInitPatchVariables(){
        for(int i=0;i<flavorsList.getLength();i++){
            Node flavorList = flavorsList.item(i);
            NodeList flavorsNodes, randomNodes;
            flavorsNodes = ((Element)flavorList).getElementsByTagName("Flavor");
            if (flavorsNodes.getLength() != 0 ) initializePatchVariables(flavorsNodes);
            randomNodes = ((Element)flavorList).getElementsByTagName("RandomFlavors");
            if (randomNodes.getLength() != 0 ) initializeRandomVariables(randomNodes);
            
        }
    }
    
    
   
    /** flavors initialisation for random definition */
    public void initializeRandomVariables(NodeList randomNodes){
        try{
            if(randomNodes.getLength()>0){
                for(int i=0; i<randomNodes.getLength(); i++){
                    /*****init patches****/
                    Element patchDescription = (Element) randomNodes.item(i);
                    String name = "RandomFlavor";
                    if (patchDescription.hasAttribute("Name")) name = patchDescription.getAttribute("Name");
                    double diffusionCoef, evaporation, quantity;
                    int nbPatches = 1;
                    if (patchDescription.hasAttribute("Number")) nbPatches = Integer.parseInt(patchDescription.getAttribute("Number"));
                    for(int j=0;j<nbPatches;j++){
                        if(patchDescription.hasAttribute("Diffusion")) diffusionCoef = Double.parseDouble(patchDescription.getAttribute("Diffusion"));
                        else diffusionCoef = Math.random();
                        if(patchDescription.hasAttribute("Evaporation")) evaporation = Double.parseDouble(patchDescription.getAttribute("Evaporation"));
                        else evaporation = Math.random();
                        if(patchDescription.hasAttribute("Quantity")) quantity = Double.parseDouble(patchDescription.getAttribute("Quantity"));
                        else quantity = Math.random()*10000;
                        PatchVariable newPatch = new PatchVariable(name+j);
                        if(patchDescription.hasAttribute("Volatile")) volatileVariables.add(newPatch);
                        newPatch.setDiffuseCoef(diffusionCoef);
                        newPatch.setEvapCoef(evaporation);
                        newPatch.setDefaultValue(quantity);
                        addPatchVariable(newPatch);
//						System.out.println(newPatch);
                    }
                }
            }
        }catch(Exception e){
            System.err.println("RandomFlavors Initialisation problem: "+ "\n" + e);
        }
    }
    
    /**
     * Initialize patch variables depending on xml parameters
     * 
     * DEPRECATED:
     * override this method is not required, but it is where you have to
     * initialize the patch variables and their properties(evaporation,
     * diffusion...) if you do not use the xml facilities :
     * Once you have created a new PatchVariable object and set
     * its properties with the methods of the class PatchVariable(setEvapCoef,
     * setDiffuseCoef and setDefaultValue) You have to add it to the simulation
     * using the addPatchVariable method:
     *
     * protected void initializePatchVariables() { PatchVariable p = new
     * PatchVariable("flavor"); p.setDiffuseCoef(0.3153); //Optional
     * p.setEvapCoef(0.025); //Optional p.setDefaultValue(32); //Optional
     * addPatchVariable(a); }
     */
    
    
    public void initializePatchVariables(NodeList flavorsNodes) {
        try{
        	int flavorNumber = 0;
            if(flavorsNodes.getLength()>0){
                for(int i=0; i<flavorsNodes.getLength(); i++){
                    /*****init patches****/
                    Element patchDescription = (Element) flavorsNodes.item(i);
                    String name = "Flavor";
                    if (patchDescription.hasAttribute("Name")) name = patchDescription.getAttribute("Name");
                    else{name+=flavorNumber; flavorNumber++;}
                    double diffusionCoef, evaporation, quantity;
                    if(patchDescription.hasAttribute("Diffusion")) diffusionCoef = Double.parseDouble(patchDescription.getAttribute("Diffusion"));
                    else diffusionCoef = Math.random();
                    if(patchDescription.hasAttribute("Evaporation")) evaporation = Double.parseDouble(patchDescription.getAttribute("Evaporation"));
                    else evaporation = Math.random();
                    if(patchDescription.hasAttribute("Quantity")) quantity = Double.parseDouble(patchDescription.getAttribute("Quantity"));
                    else quantity = Math.random()*10000;
                    PatchVariable newPatch = new PatchVariable(name);
                    if(patchDescription.hasAttribute("Volatile")) volatileVariables.add(newPatch);
                    newPatch.setDiffuseCoef(diffusionCoef);
                    newPatch.setEvapCoef(evaporation);
                    newPatch.setDefaultValue(quantity);
                    addPatchVariable(newPatch);
//                  System.out.println(newPatch);
                }
            }
            
        }catch(Exception e){
            System.err.println("Flavors Initialisation problem: "+ "\n" + e);
        }
    }
    
    
    /** genetic engine initialization. The genetic engine management is given to a metamanager */
    public void initializeGenetic()
    {
    	if(metamanager == null)	metamanager = new MetaManager(this, poolsList, managersList, flavors, defaultRefresh);
    	else {
    		//metamanager.clear();
    		killAgent(metamanager);
    		metamanager = new MetaManager(this, poolsList, managersList, flavors, defaultRefresh);
    	}
    	
    	addObserver(metamanager,false, "MetaManager");
    }
    
    
    /////////////////////////////////////////// Activation and running methods
    /** MadKit usage */
    public void activate() {
        println("activated");
		if (onScreen!=null) {
			onScreen.initialisation();
		}        
		// println("activated");
        //phase 0: getting simulation parameters if a xml file exists
        loadConfigFile();
        initializeNodes();
        initializeSimulationParameters();
        //phase 1: creating an organization
        int i = 2;
        if (isGroup(COMMUNITY,simulationName)) {
            while (isGroup(COMMUNITY,simulationName + " " + i))
                i++;
            simulationName += " " + i;
        }
        createGroup(false, COMMUNITY,simulationName, null, null);
        requestRole(COMMUNITY,simulationName, "launcher",null);
		if (onScreen!=null) {
			onScreen.setName(simulationName);
		}        
		//phase 2:initializing simulation
        launchScheduler(new TurtleScheduler(simulationName));
        thisInitPatchVariables();
        launchedObservers=new ArrayList<AbstractAgent>();
        if(geneticMode) initializeGenetic();
        createLogoWorld();
        if(simulationFilePath==null)
            addSimulationAgents();
        else{
            initialPopulation=new ArrayList<Turtle>();
            thisCreateInitPopulation();
            thisInitObservers();
            thisInitViewers();
        }
        println("Simulation is initialized !!");
        println("Click to begin...");
        while (true) {
            pause(50);
            if (start) {
                break;
            }
            if (nextMessage() != null) {
                onScreen.startStop.doClick();
            }
        }
        
        //phase 3: launching simulation
        start = true;
        run = true;
        sendMessage(sch.getAddress(), new TopMessage(TurtleScheduler.RUNNING));
        if(!standAloneMode) disposeMyGUI();
    }
    
    /** MadKit kernel usage */
    public final void live() {
                /*
                 * if(pythonOn) { AbstractAgent a = new
                 * PythonCommandCenter(simulationName); launchAgent(a, "Python command
                 * center",true); launchedAgents.add(a); }
                 */
        int i = -1;
        while (true) {
            pause(cycleDisplayEvery);
            if (sch != null && start && i != sch.iteration) {
                i = sch.iteration;
                println("step " + i);
            }
        }
        
    }
    
    /** MadKit kernel usage. No redefinition. Closing the simulation. */
    public final void end() {
        println("Closing simulation");
        println("Please wait...");
        sendMessage(sch.getAddress(), new TopMessage(-1));
        killAgent(sch);
        if (playGround != null)
            killAgent(playGround);
        for (Iterator<AbstractAgent> e = launchedObservers.iterator(); e.hasNext();)
            killAgent(e.next());
        leaveGroup(simulationName);
        System.gc();
        System.runFinalization();
    }
    
    
    final public LauncherGui getOnScreen(){
        return onScreen;
    }
    
    //
    final public void setWidth(int add) {
        envWidth = add;
    }
    final public int getWidth() {
        return envWidth;
    }
    
    final public void setCellSize(int add) {
        cellSize = add;
    }
    final public int getCellSize() {
        return cellSize;
    }
    
    final public void setHeight(int add) {
        envHeight = add;
    }
    final public int getHeight() {
        return envHeight;
    }
    
    final public String getSimulationName() {
        return simulationName;
    }
    final public void setSimulationName(String name) {
        simulationName = name;
    }
    
    final public void setCyclePause(int add) {
        cyclePause = add;
        if (sch != null)
            sch.delay = cyclePause;
    }
    
    /** reseting the simulation. Unstable in genetic Mode*/
    final public void setReset() {
        sendMessage(sch.getAddress(), new TopMessage(TurtleScheduler.STOPED));
        start = false;
        println("Reseting: Please wait ...");
        /*for (Iterator e = launchedObservers.iterator(); e.hasNext();){
        	System.out.println("Kill :" + ((AbstractAgent) e.next()).getAddress());
            killAgent((AbstractAgent) e.next());
        }*/
        run = false;
        playGround.clearAllTurtles();
        playGround.initGrid();
        initializeNodes();
        initializeSimulationParameters();
        killAgent(playGround);
        createLogoWorld();//TODO methode voir si param chang
        flavors = null;
        thisInitPatchVariables();
        launchedObservers.clear();
        if(geneticMode) initializeGenetic();
        if(simulationFilePath==null)
            addSimulationAgents();
        else{
            initialPopulation=new ArrayList<Turtle>();
            thisCreateInitPopulation();
            launchedObservers=new ArrayList<AbstractAgent>();
            thisInitObservers();
            thisInitViewers();
        }
        start=true;
        run=true;
        sendMessage(sch.getAddress(), new TopMessage(TurtleScheduler.RUNNING));
    }
    
    
    /** Specifically for genetic engine usage. Reset the simulation by replacing agents and flavors */
    public void newPopulation(){
    	if(!geneticMode) setReset();
		else{
			sendMessage(sch.getAddress(), new TopMessage(TurtleScheduler.STOPED));
	        start = false;
	        println("Reseting: Please wait ...");
	        run = false;
	        playGround.clearAllTurtles();
	        playGround.clearVariables(volatileVariables);
	        /*//Deprecated
            playGround.initGrid();
	        initializeNodes();
	        initializeSimulationParameters();
	        killAgent(playGround);
	        createLogoWorld();//TODO methode voir si param chang
	        flavors = null;
	        thisInitPatchVariables();
	        if(geneticMode) initializeGenetic();
	        launchedObservers.clear();*/
	        if(simulationFilePath==null)
	            addSimulationAgents();
	        else{
	            initialPopulation=new ArrayList<Turtle>();
	            thisCreateInitPopulation();
	            /*//Deprecated
	            launchedObservers=new ArrayList();
	            thisInitObservers();
	            thisInitViewers();*/
	        }
	        start=true;
	        run=true;
	        sendMessage(sch.getAddress(), new TopMessage(TurtleScheduler.RUNNING));
		}
    }
    
    /** setter for toroidal world usage */
    final public void setWrapModeOn(boolean b) {
    	/*
    	 *  From greg: j'ai ajoute le if (run). ca sert a rien 
    	 * de pauser si y a pas de scheduler ou si c pas run 
    	 * (ca produit des exceptions a l'init)
    	 */
    	if(run)
    		pauseSimulation();
        wrap = b;
        if (playGround != null){
            playGround.wrap = b;
            if (diffusion)
                playGround.initNeighborhood();
        }
        if(run)
            sendMessage(sch.getAddress(), new TopMessage(TurtleScheduler.RUNNING));
    }
    
    void setStop() {
        if (run) {
            run = false;
            pauseSimulation();
            println("Simulation paused");
        } else {
            sendMessage(sch.getAddress(), new TopMessage(TurtleScheduler.RUNNING));
            println("Simulation running");
            run = true;
        }
    }
    
    final public void stepByStep() {
        sendMessage(sch.getAddress(), new TopMessage(TurtleScheduler.STEP));
        run=false;
    }
    
    // ////////////////////////////////////////////////////////////////
    final void initializeVariables() {
        for (Enumeration<PatchVariable> e = flavors.elements(); e.hasMoreElements();) {
            PatchVariable f = e.nextElement();
            playGround.addGridVariable(f);
        }
        playGround.initNeighborhood();
    }
    
    final void launchSimulation() {
        println("Launching simulation !");
        println("Please wait...");
        /*initializePatchVariables();
        createLogoWorld();
        addSimulationAgents();*/
        sendMessage(sch.getAddress(), new TopMessage(TurtleScheduler.RUNNING));
    }
    
    final void createLogoWorld() {
        playGround = new TurtleEnvironment(envWidth, envHeight, simulationName);
        playGround.wrap = wrap;
        launchAgent(playGround, simulationName + " world", false);
        if (flavors != null)
            initializeVariables();
    }
    
    final void initViewer(Viewer v, int cellS, String viewerName) {
        preferedViewer = v.getClass();
        v.cellSize = cellS;
        addObserver(v, true, viewerName);
        /*if (start && run) {
            sendMessage(sch.getAddress(), new TopMessage(TurtleScheduler.));
            waitNextMessage();
            sendMessage(sch.getAddress(), new TopMessage(TurtleScheduler.));
        }*/
    }
    
    // ///////////////////////////////////////////////////////////////////////////////
    /** place a turtle at a random patch */
    final protected void addTurtle(Turtle t) {
    	if(geneticMode) {
    		metamanager.manage((GTurtle)t);
    		((GTurtle)t).setFlavors(flavors);
    	}
        playGround.addAgent(t);
    }
    
    /**
     * place a turtle on the patch (u,v). Be sure to use the addTurtle methods
     * in the addSimulationAgents method
     */
    final protected void addTurtle(Turtle t, int u, int v) {
    	if(geneticMode) {
    		metamanager.manage((GTurtle)t);
    		((GTurtle)t).setFlavors(flavors);
    	}
        playGround.addAgent(t, u, v);
    }
    
    /**
     * Add a specified Observer to the simulation. Be careful, use this method
     * only in the addSimulationAgents method.
     */
    final public void addObserver(Observer theObserver, boolean hasGUI, String agentName) {
        theObserver.simulationGroup = simulationName;
        theObserver.envWidth = envWidth;
        theObserver.envHeight = envHeight;
        theObserver.setFlavors(flavors);
        launchAgent(theObserver, agentName, hasGUI);
        launchedObservers.add(theObserver);
    }
    
    /**
     * Add a specified Observer to the simulation. Be careful, do not use these
     * methods before the addSimulationAgents method has been invoked. (during
     * life cycle it is ok)
     */
    final public void addObserver(Observer theObserver, boolean hasGUI) {
        addObserver(theObserver, hasGUI, simulationName + " Observer");
    }
    
    /**
     * Add a default world viewer with the current cell size. Be careful, do not
     * use these methods before the addSimulationAgents method has been invoked.
     * (during life cycle it is ok)
     */
    final public void addViewer() {
        addViewer(cellSize);
    }
    
    /** add a default world viewer with the specified cell size: cellS */
    final public void addViewer(int cellS) {
        try{
            addViewer(preferedViewer.newInstance(), cellS);
        } catch(Exception e){
            //println("unable to add the last viewer: "+e);
            //e.printStackTrace(System.err);
            addViewer(new Viewer(),cellS);
        }
    }
    
    /** add a specific world viewer with the specified cell size: cellS */
    final public void addViewer(Viewer v, int cellS, String viewerName) {
        initViewer(v, cellS, viewerName);
    }
    /** add a specific world viewer with the specified cell size: cellS */
    final public void addViewer(Viewer v, int cellS) {
        initViewer(v, cellS, simulationName + " Observer");
    }
    /** add a specific world viewer with the current cell size */
    final public void addViewer(Viewer v, String viewerName) {
        initViewer(v, cellSize, viewerName);
    }
    /** add a specific world viewer with the current cell size */
    final public void addViewer(Viewer v) {
        addViewer(v, cellSize);
    }
    /**
     * specify which scheduler must be used : this method is called when the launcher is activated, in the activate method
     * you can add a specific scheduler that overrides the scheduleWorld method for
     * example)
     */
    protected void launchScheduler(TurtleScheduler s) {
        sch = s;
        sch.group = simulationName;
        launchAgent(sch, simulationName + " scheduler", false);
        waitNextMessage();
    }
    
    /**
     * Be careful, use this method only in the initializeSimulation method. This
     * method add a patch variable (a PatchVariable Object) defined with the
     * PatchVariable constructor and the set methods
     */
    final protected void addPatchVariable(PatchVariable variable) {
        if (flavors == null)
            flavors = new Vector<PatchVariable>();
        flavors.addElement(variable);
    }
    
    /**
     * Override this method is compulsory (abstract). It is in this method that
     * the optional agents of the simulation (turtles, viewers and observers)
     * have to be added. To add these agents you have to use the "add" methods
     * of a Launcher: - addTurtle - addViewer - addObserver
     */
    public void addSimulationAgents(){};
    
    // ///////////////////////////////////////////////////////////////////////////
    /** MadKit kernel usage */
    final public void initGUI() {
    	onScreen = new LauncherGui(this);
        if(standAloneMode) setGUIObject(onScreen);
        
    }
    
    
    public void launchPython() throws Exception {
        AbstractAgent a = new PythonCommandCenter(simulationName);
        launchAgent(a, "Python command center", true);
        launchedObservers.add(a);
    }
    
    public void stopOrResumeSimulation() {
        onScreen.startStop.doClick();
    }
    
    public void pauseSimulation(){
    	sendMessage(sch.getAddress(), new TopMessage(TurtleScheduler.PAUSED));
        waitNextMessage();
    }

	public void runSimulation() {
		// TODO Auto-generated method stub
		
	}

	public void stopSimulation() {
		// TODO Auto-generated method stub
		
	}
    
}
