/*
 * Genetic Package Copyright (C) 2002-2007 Gregory Beurier
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or any later version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package turtlekit2.genetic;

import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.InputEvent;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;

import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;

/**
 * <p>Title : The Genome GUI. </p>
 * <p>Description : It provides edition, deletion, copy, modifications, mutations, etc.. of genes and genomes in real time during the simulation.</p>
 * @author Gregory Beurier
 */  
public class GenomeGUI extends JPanel {
	ExtendedJPopupMenu genePopupMenu;

	Genome myGenome;

	public GenomeGUI(Genome genome) {
		myGenome = genome;
		refresh();
	}

	public void createPopupMenu(){
		genePopupMenu = new ExtendedJPopupMenu();

		JMenuItem saveGene = new JMenuItem("Save");
		JMenuItem replaceGene = new JMenuItem("Replace");
		JMenuItem resetGene = new JMenuItem("Reset");
		JMenuItem setFixed = new JMenuItem("Set \"Fixed\"");
		JMenuItem setUnFixed = new JMenuItem("Set \"UNFixed\"");
		JMenuItem removeGene = new JMenuItem("Remove");
		JMenuItem setName = new JMenuItem("Change Name");
		JMenuItem editGene = new JMenuItem("Edit BitGene");
		JMenuItem editGene2 = new JMenuItem("Edit Codon");
		JMenuItem editGene3 = new JMenuItem("Edit Codons");
		genePopupMenu.add(saveGene);
		genePopupMenu.add(replaceGene);
		genePopupMenu.add(resetGene);
		genePopupMenu.add(setFixed);
		genePopupMenu.add(setUnFixed);
		genePopupMenu.add(removeGene);
		genePopupMenu.add(setName);
		genePopupMenu.add(editGene);
		genePopupMenu.add(editGene2);
		genePopupMenu.add(editGene3);
		
		saveGene.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) {
				saveGeneEvent(evt);
			}
		});

		replaceGene.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) {
				replaceGeneEvent(evt);
			}
		});
		
		resetGene.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) {
				resetGeneEvent(evt);
			}
		});
		
		setFixed.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) {
				setFixedEvent(evt);
			}
		});
		
		setUnFixed.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) {
				setUnFixedEvent(evt);
			}
		});

		removeGene.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) {
				removeGeneEvent(evt);
			}
		});
		
		editGene.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) {
				editGeneEvent(evt);
			}
		});
		
		editGene2.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) {
				editGeneEvent2(evt);
			}
		});
		
		editGene3.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) {
				editGeneEvent3(evt);
			}
		});
		
		setName.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) {
				setNameEvent(evt);
			}
		});
	}
	
	public void createPanelContent(){
		this.setLayout(new GridLayout(myGenome.getSize(), 4, 45, 5));
		for (int i = 0; i < myGenome.getSize(); i++) {
			ExtendedJButton geneButton = new ExtendedJButton("Gene "+ new Integer(i).toString(), myGenome.getGene(i), i);
			geneButton.addMouseListener(new MouseListener() {
				public void mouseClicked(MouseEvent m) {}
				public void mouseEntered(MouseEvent e) {}
				public void mousePressed(MouseEvent m) {
					if ((m.getModifiers() & InputEvent.BUTTON3_MASK) != 0) {
						genePopupMenu.show(m.getComponent(), m.getX(), m.getY(), ((ExtendedJButton) (m.getSource())).getGene());
					}
				}
				public void mouseReleased(MouseEvent m) {
					if ((m.getModifiers() & InputEvent.BUTTON3_MASK) != 0) {
						genePopupMenu.show(m.getComponent(), m.getX(), m.getY(), ((ExtendedJButton) (m.getSource())).getGene());
					}
				}

				public void mouseExited(MouseEvent e) {}
			});
			JLabel geneString = new JLabel(myGenome.getGene(i).toString());
			JLabel geneName = new JLabel(myGenome.getGene(i).getName());
			JLabel geneFixed = new JLabel(new Boolean(myGenome.getGene(i).isFixed()).toString());
			this.add(geneButton);
			this.add(geneName);
			this.add(geneFixed);
			this.add(geneString);
		}
	}
	
	public void refresh(){
		this.removeAll();
		createPanelContent();
		createPopupMenu();
	}
	
	public Gene getSelectedGene(ActionEvent evt){
		JMenuItem me = (JMenuItem) (evt.getSource());
		ExtendedJPopupMenu parentMenu = (ExtendedJPopupMenu) (me.getParent());
		return ((Gene) (parentMenu.getObject()));
	}
	
	public void saveGeneEvent(ActionEvent evt) {
		GeneticFileManager.saveGene(this,getSelectedGene(evt));
	}

	public void replaceGeneEvent(ActionEvent evt) {
		Gene newGene = GeneticFileManager.loadGene(this);
		if(newGene != null){
			int index = myGenome.indexOf(getSelectedGene(evt));
			myGenome.remove(getSelectedGene(evt));
			myGenome.add(index, newGene);
			refresh();
		}
	}

	public void resetGeneEvent(ActionEvent evt) {
		getSelectedGene(evt).init();
		refresh();
	}
	
	public void setFixedEvent(ActionEvent evt) {
		getSelectedGene(evt).setFixed(true);
		refresh();
	}
	
	public void setUnFixedEvent(ActionEvent evt) {
		getSelectedGene(evt).setFixed(false);
		refresh();
	}
	
	public void removeGeneEvent(ActionEvent evt) {
		myGenome.remove(getSelectedGene(evt));
		refresh();
	}
	
	public void setNameEvent(ActionEvent evt) {
		String inputValue = JOptionPane.showInputDialog("Please input the new Gene Name");
		getSelectedGene(evt).setName(inputValue);
		refresh();
	}

	/** Standard binary edition */
	public void editGeneEvent(ActionEvent evt) {
		String inputValue = JOptionPane.showInputDialog("Please input the new Gene Value");
		for(int i=0; i<((BitGene)((BitGeneList)(getSelectedGene(evt))).get(1)).getSize(); i++){
			if(inputValue.charAt(i) == '1') ((BitGene)((BitGeneList)(getSelectedGene(evt))).get(1)).set(i);
			else ((BitGene)((BitGeneList)(getSelectedGene(evt))).get(1)).set(i, false);
		}
		refresh();
	}
	
	/** Edit one part (the codon) of the binary */
	public void editGeneEvent2(ActionEvent evt) {
		String inputValue = JOptionPane.showInputDialog("Please input the codon size (in bits)");
		String inputValue2 = JOptionPane.showInputDialog("Please input the codon number (in integer)");
		String inputValue3 = JOptionPane.showInputDialog("Please input the codon value (in integer)");
		((BitGene)(getSelectedGene(evt))).setCodonValue(Integer.parseInt(inputValue),Integer.parseInt(inputValue2),Integer.parseInt(inputValue3));
		refresh();
	}
	
	/** Edit all codons of the genes. The codons are defined by their size (the number of bits of the codon) */
	public void editGeneEvent3(ActionEvent evt) {
		int codonNumber = 0;
		String inputValue = JOptionPane.showInputDialog("Please input the codons size (in bits)");
		String inputValue3 = JOptionPane.showInputDialog("Please input the codon value (in integer)");
		((BitGene)(getSelectedGene(evt))).setCodonValue(Integer.parseInt(inputValue),codonNumber,Integer.parseInt(inputValue3));
		while(Integer.parseInt(inputValue3) != -1){
			codonNumber++;
			inputValue3 = JOptionPane.showInputDialog("Please input the codon value for the " + codonNumber + "th codon");
			((BitGene)(getSelectedGene(evt))).setCodonValue(Integer.parseInt(inputValue),codonNumber,Integer.parseInt(inputValue3));
		}
		refresh();
	}
	
}