/*
* Genetic Package for TurtleKit 2: TurtleKit - A 'reactive simulation platform' using MadKit Kernel
* Copyright (C) 2002-2007 Gregory Beurier
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

package turtlekit2.genetic;



import java.util.BitSet;
import java.util.Random;

import turtlekit2.kernel.XMLAttributes;

/**
 * <p>Title : BitGene </p>
 * <p>Description : A basic Gene encoded in a bitset. Given methods are about interpretation, reading,
 * crossing, mutation of the gene </p>
 * <p></p>
 * <p>XML Attributes :</p>
 * <p></p>
 * <p>Size: the size of the BitSet. default is 5</p>
 * <p>Name: the name of the BitGene. default is BitGene</p>
 * <p>Filling: the percentage of 1 in the BitGene. default is 50</p>
 * <p>CrossMethod: the method of crossing 2 genes. Value: 1 for one point crossing, 2 for two points crossing. default is twoPointCrossing</p>
 * <p>MutationRate: the mutation rate of the gene expressed beetween 2 generations in 1/10000. default is 9900.<p>
 * @author Gregory Beurier
 */  

public class BitGene extends BitSet implements Gene {

	final static int ONE_POINT_CROSSING = 1;

	final static int TWO_POINT_CROSSING = 2;

	String name;

	boolean fixed = false;
	
	XMLAttributes attrib = new XMLAttributes();

	Random randomizer = new Random();

	int size;

	/**
	 * Constructors
	 */
	public BitGene() {
	}

	public BitGene(String n) {
		name = n;
	}

	public void createFrame() {

	}

	/** initialisation method. Default size = 5, default name = BitGene, default filling rate = 50% */
	public void init() {
		double fillingRate = 50;
		if(attrib.containsKey("Size")) size = attrib.getInt("Size"); else size = 5;
		if(attrib.containsKey("Name")) name = attrib.getString("Name"); else name = "BitGene";
		if(attrib.containsKey("Filling")) fillingRate = attrib.getInt("Filling"); else fillingRate = 50;
		
		for (int i = 0; i < size; i++) {
			if ((Math.random() * 100) > fillingRate) {
				set(i);
			}
		}
	}

	/** copy method using logical OR on the bitset */
	public void copy(Gene pattern) {
		name = pattern.getName();
		attrib = pattern.getAttrib();
		size = ((BitGene) pattern).getSize();
		this.clear();
		this.or((BitGene) pattern);
	}

	/** copy method using logical OR on the bitset */
	public void copyBitSet(BitSet pattern) {
		this.clear();
		this.or(pattern);
	}
	
	/** equals method */
	public boolean isEqual(Gene gene){
		if(((BitGene)gene).length() != this.length()) return false;
		for(int i=0; i<((BitGene)gene).length();i++){
			if(get(i) != ((BitGene)gene).get(i)) return false; 
		}
		return true;
	}

	/** crossing methods for reproduction using bitset masks. default is two point crossing */
	public void cross(Gene fatherGene, Gene motherGene) {
		int method = TWO_POINT_CROSSING;
		if(attrib.containsKey("CrossMethod")) method = attrib.getInt("crossMethod");
		
		if (method == ONE_POINT_CROSSING) {
			int cutIndex = randomizer.nextInt(size);
			BitSet maskLeft = new BitSet(size);
			BitSet maskRight = new BitSet(size);
			maskLeft.set(0, cutIndex);
			maskRight.set(cutIndex, size);
			if (randomizer.nextBoolean()) {
				maskLeft.and((BitSet) motherGene);
				maskRight.and((BitSet) fatherGene);
				maskLeft.or(maskRight);
				copyBitSet(maskLeft);
			} else {
				maskLeft.and((BitSet) fatherGene);
				maskRight.and((BitSet) motherGene);
				maskLeft.or(maskRight);
				copyBitSet(maskLeft);
			}
		}
		if (method == TWO_POINT_CROSSING) {
			int cutIndex1 = randomizer.nextInt(size / 2);
			int cutIndex2 = size - (randomizer.nextInt(size / 2));
			BitSet maskExtern = new BitSet(size);
			BitSet maskIntern = new BitSet(size);
			maskExtern.set(0, cutIndex1);
			maskIntern.set(cutIndex1, cutIndex2);
			maskExtern.set(cutIndex2, size);
			if (randomizer.nextBoolean()) {
				maskExtern.and((BitSet) motherGene);
				maskIntern.and((BitSet) fatherGene);
				maskExtern.or(maskIntern);
				copyBitSet(maskExtern);
			} else {
				maskExtern.and((BitSet) fatherGene);
				maskIntern.and((BitSet) motherGene);
				maskExtern.or(maskIntern);
				copyBitSet(maskExtern);
			}
		}
	}

	
	/** Mutation method. Given a chance of n/1000 to switch a bit of the gene. default n is 10000-9900 */
	public void mutate() {
		double mutationRate = 9900;
		if(attrib.containsKey("MutationRate")) mutationRate = attrib.getDouble("MutationRate");
		for (int j = 0; j < size; j++) {
			if (randomizer.nextInt(10000) > mutationRate) {
				flip(j);
			}
		}
	}

	/** toString method. Display the bitgene in integer. */
	public String toString() {
		String bitRepresentation = (name + ":");
		for (int i = 0; i < getSize(); i++) {
			bitRepresentation += intValue(i);
		}
		bitRepresentation += "  ";
		return bitRepresentation;
	}

	/** Int value calculation of a bit */
	public int intValue(int i) {
		if (get(i)) {
			return 1;
		} else {
			return 0;
		}
	}

	/** Int value calculation of a group of bits*/
	public int intValue(int fromIndex, int toIndex) {
		int result = 0;
		for (int k = fromIndex; k < toIndex; k++) {
			result += intValue(k) * Math.pow(2, (toIndex - fromIndex) - (k - fromIndex + 1));
		}
		return result;
	}

	/** Int value calculation of the bitgene*/
	public int intValue() {
		int result = 0;
		for (int i = 0; i < size; i++) {
			result += intValue(i) * Math.pow(2, size - (i + 1));
		}
		return result;
	}

	public void setIntValue(int value) {
		System.out.println("ok");
		char tab[] = Integer.toBinaryString(value).toCharArray();
		clear();
		if (tab.length > size)
			set(0, size);
		else {
			for (int i = 0; i < tab.length; i++) {
				if (tab[i] == '1') {
					set(size - tab.length + i);
				}
			}
		}
	}

	/** set the value of a groupe of bits (a codon). Codon number represents the position of the codon in the gene */
	public void setCodonValue(int codonSize, int codonNumber, int codonValue) {
		char tab[] = Integer.toBinaryString(codonValue).toCharArray();
		int fromIndex = codonNumber*codonSize;
		int toIndex = codonNumber*codonSize+codonSize;
		clear(fromIndex, toIndex);
		if (tab.length > codonSize)
			set(fromIndex, toIndex);
		else {
			for (int i = 0; i < tab.length; i++) {
				if (tab[i] == '1') {
					set(toIndex - tab.length + i);
				}
			}
		}
	}
	
	public String getName() {
		return name;
	}

	public void setName(String name) {
		this.name = name;
	}

	public XMLAttributes getAttrib() {
		return attrib;
	}

	public void setAttrib(XMLAttributes geneAttribute) {
		attrib = geneAttribute;
	}

	public String cardinalityString() {
		String bitRepresentation = new Integer(cardinality()).toString();
		return bitRepresentation;
	}

	public int getSize() {
		return size;
	}

	public void setSize(int size) {
		this.size = size;
	}

	public int size() {
		return getSize();
	}
	/**
	 * @return Returns true if the bitgene cannot be mutated or crossed.
	 */
	public boolean isFixed() {
		return fixed;
	}
	/**
	 * @param Set the mutation/crossing behavior of the gene.
	 */
	public void setFixed(boolean fixed) {
		this.fixed = fixed;
	}
}
