/*
 * Organization.java - Kernel: the kernel of MadKit
 * Copyright (C) 1998-2008 Olivier Gutknecht, Fabien Michel, Jacques Ferber
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package madkit.kernel;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;

/** This class is the heart of AGR mechanisms of MadKit 3.0. It implements the Aalaadin mechanisms
with new security possibilities using groups and role
    @author Fabien Michel
    @version 1.0 MadKit 3.0
    @since MadKit 3.0
 */
@SuppressWarnings("serial")
final class Organization extends HashMap <String, Group>
{

	private HashSet<AgentAddress> agentsIn = new HashSet<AgentAddress>();

///////////////////////////////////// 	FUNDAMENTAL OPERATIONS : CREATE, REQUESTRole, LEAVEROLE & LEAVEGROUP

	synchronized boolean createGroup(AgentAddress creator,boolean distributed,String groupName, String description, GroupIdentifier theIdentifier)
	{
		// the group does not exist
		if(! containsKey(groupName))
		{
			groupName.intern();
			put(groupName,new Group(creator, distributed, description, theIdentifier));
			return true;
		}
		else
			if ( get(groupName).isEmpty())	// the group is overlooked but is not registered as a real one
			{
				get(groupName).realCreation(creator, distributed, description, theIdentifier);
				return true;
			}
		return false;	// the group already exists
	}

	synchronized void requestRole(AgentAddress requester, String groupName,String roleName,Object memberCard) throws CGRException
	{
		final Group theGroup = get(groupName);
		try {
			if((! theGroup.isEmpty()) && (! roleName.equals("group manager")) ) // TODO find another way to manage "group manager"
				theGroup.requestRole(requester,roleName,memberCard);
			agentsIn.add(requester);
		} catch (NullPointerException e) {
			throw new CGRException(-2,null,groupName,null,e);	// the Group does not exist
		} catch (CGRException e) {
			e.setGroup(groupName);
			throw e;
		}
	}

//	backup
//	synchronized int requestRole(AgentAddress requester, String groupName,String roleName,Object memberCard)
//	{
//	final Group theGroup = get(groupName);
//	if(theGroup != null && (! theGroup.isEmpty()) && (! roleName.equals("group manager")) )
//	return theGroup.requestRole(requester,roleName,memberCard);
//	return -3;	// the Group does not exist
//	}

	synchronized boolean leaveGroup(AgentAddress requester, String groupName) throws CGRException
	{
		try {
			if(get(groupName).leave(requester) )
				remove(groupName);
		} catch (NullPointerException e) {
			throw(new CGRException(-2,null,groupName,null,e)); // the group does not exist
		} catch (NotAMemberException e) {
			throw(new CGRException(-7,null,groupName,null,e)); // not a member
		}
		return isEmpty();
	}

	synchronized boolean leaveRole(AgentAddress requester, String groupName,String roleName) throws CGRException
	{
		try {
			if( get(groupName).leaveRole(requester, roleName) )
				remove(groupName);
		} catch (NullPointerException e) {
			throw(new CGRException(-2,null,groupName,null,e));
		} catch (CGRException e) {
			e.setGroup(groupName);
			throw(e);
		}
		return isEmpty();
	}

///////////////////////////////////////////////////////////////////////////////////////////////

/////////////////////////////////// OVERLOOKER OPEARTIONS
	synchronized boolean addOverlooker(AgentAddress requester, Overlooker o, Object accessCard)
	{
		try {
			return get(o.group).addOverlooker(requester, o, accessCard);
		} 
		// the group does not exist yet
		catch (NullPointerException e) {
			Group theGroup = new Group();
			put(o.group,theGroup);
			return theGroup.addOverlooker(requester, o, accessCard);
		}
	}

	synchronized boolean removeOverlooker(Overlooker o)
	{
		try {
			if(get(o.group).removeOverlooker(o)){
				remove(o.group);
				return isEmpty();
			}
		}
		catch (NullPointerException e) {
		}
		return false;
	}

///////////////////////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////ORGANIZATION INFO OPERATIONS
	boolean isPlayingRole(AgentAddress theAgent, String groupName, String roleName)
	{
		try {
			return get(groupName).isPlayingRole(theAgent, roleName);
		} catch (NullPointerException e) {
			return false;
		}
	}

	/**
	 * Checks if the group exists and has members (not a virtual one) 
	 * 
	 * @param groupName the group name
	 * 
	 * @return true, if is group
	 */
	Group getNotEmptyGroup(final String groupName) //TODO check speed with and without try
	{
		final Group g = get(groupName);
		try {
			if(! g.isEmpty())
				return g;
		} catch (NullPointerException e) {
		}
		return null;
	}

	Collection<AgentAddress> getRolePlayers(String groupName, String roleName) throws CGRException
	{
		try {
			return getNotEmptyGroup(groupName).getRolePlayers(roleName);
		} 
		catch (NullPointerException e) {
			throw new CGRException(-2,null,groupName,null,e);
		}
		catch (CGRException e) {
			e.setGroup(groupName);
			throw(e);
		}
	}

//	backup
//	Collection<AgentAddress> getRolePlayers(String groupName, String roleName)
//	{
//	try{
//	return get(groupName).getRolePlayers(roleName);
//	}
//	catch (NullPointerException e) {
//	}
//	return null;
//	}


	AgentAddress getRolePlayer(String groupName, String roleName) throws CGRException {
		try {
			return getNotEmptyGroup(groupName).getRolePlayer(roleName);
		} catch (CGRException e) {
			e.setGroup(groupName);
			throw(e);
		} catch (NullPointerException e) {
			throw new CGRException(-2,null,groupName,null,e);
		}
	}

	String[] getGroupRolesOf(AgentAddress agent, String groupName){
		try{
			return get(groupName).getRolesOf(agent);
		}
		catch (NullPointerException e) {
			return new String[0];
		}
	}

	boolean isDistributed(String groupName)
	{
		try {
			return Kernel.onLineMode && get(groupName).isDistributed();
		} catch (NullPointerException e) {
			return false;
		}
	}

	String[] getRolesIn(String groupName)
	{
		try {
			return get(groupName).availableRoles();
		} catch (NullPointerException e) {
			return new String[0];
		}
	}

	synchronized String[] getGroups()
	{
		Collection<String> c = new HashSet<String>(size());
		for (Map.Entry<String,Group> e : entrySet())
		{
			if( ! e.getValue().isEmpty())
				c.add(e.getKey());
		}
		return c.toArray(new String[c.size()]);
	}

	synchronized Collection<String> getCurrentGroupsOf(AgentAddress agent)
	{
		Collection<String> c = new HashSet<String>(size());
		for (Map.Entry<String,Group> e : entrySet()) {
			if( e.getValue().isPlayingRole(agent, Group.MEMBER_DEFAULT_ROLE) )
				c.add(e.getKey());
		}
		return c;
	}


	synchronized Map getLocalOrganization()
	{
		Map groupsName = new Hashtable();
		for (Map.Entry<String,Group> e : entrySet())
		{
			if (! e.getValue().isEmpty())
				groupsName.put(e.getKey(),e.getValue().mapForm());
		}
		return groupsName;
	}


///////////////////////////////////////////////		IMPORT EXPORT

	synchronized Organization exportOrg()
	{
		Organization org = new Organization();
		for (Map.Entry<String,Group> e : entrySet())
		{
			Group g = e.getValue();
			if(g.isDistributed() && ! g.isEmpty())
				org.put(e.getKey(),e.getValue());
		}
		return org;
	}

	synchronized void importOrg(Organization org, boolean priority) {
		for (Map.Entry<String,Group> e : org.entrySet()) {
			if (! containsKey(e.getKey())) {
				Group newGroup = new Group();
				put(e.getKey(),newGroup);
				newGroup.merge(e.getValue(),priority);
			}
			else
				get(e.getKey()).merge(e.getValue(),priority);
		}
	}

	/**
	 * Removes the agents which are from a specific distant kernel.
	 * 
	 * @param distantAddress the distant address
	 * 
	 * @return true, if the organization is empty after the operation
	 */
	synchronized boolean removeAgentsFromKernel(KernelAddress distantAddress) {
		for(final Iterator<Group> i = values().iterator();i.hasNext();){
			final Group g = i.next();
			if(g.isDistributed() && g.removeAgentsFromKernel(distantAddress))
				i.remove();
//			try {
//			for(AgentAddress groupMember : g.getRolePlayers(Group.MEMBER_DEFAULT_ROLE))
//			try {
//			if(groupMember.getKernel().equals(distantAddress) && g.leave(groupMember)) 
//			i.remove();
//			} catch (Exception e) {
//			System.err.println("!!!!!! This should not happen : kernel internal error on group :"+g+" with agent "+groupMember);
//			e.printStackTrace();
//			}
//			}
//			catch (/*CGRNotFound*/Exception e) {
//			System.err.println("!!!!!! This should not happen : kernel internal error on group :"+g);
//			e.printStackTrace();
//			}
		}
		return isEmpty();
	}

	ArrayList<String> removeAgentFromAllGroups(AgentAddress theAgent) {
		ArrayList<String> groups = new ArrayList<String>();
		for (Iterator<Map.Entry<String, Group>> e = this.entrySet().iterator();e.hasNext();) {
			Map.Entry<String, Group> entry = e.next();
			Group g = entry.getValue();
			if(g.removeAgentFromGroup(theAgent)){
				groups.add(entry.getKey());
				if(g.isEmpty())
					e.remove();
			}
		}
		return groups;

	}

	boolean agentIn(AgentAddress theAgent){
		return agentsIn.contains(theAgent);
	}

	void removeAgentFromGroups(Kernel kernel, AgentAddress theAgent, String orgName) {
		for (Map.Entry<String, Group> entry : new HashMap<String, Group>(this).entrySet()) {
			entry.getValue().removeAgent(kernel,theAgent,orgName,entry.getKey());
		}

	}

	/**
	 * Removes all the agents belonging to distant kernels.
	 * 
	 * @return true, if the organization is empty after the operation
	 */
//	synchronized boolean removeDistantAgents() {
//	for(Iterator<Group> i = values().iterator();i.hasNext();) {
//	Group g = i.next();
//	if(g.isDistributed())
//	try {
//	for(AgentAddress groupMember : g.getRolePlayers(Group.MEMBER_DEFAULT_ROLE))
//	try {
//	if( (! groupMember.isLocal()) && g.leave(groupMember)) 
//	i.remove();
//	} catch (NotAMemberException e) {
//	System.err.println("!!!!!! This should not happen : kernel internal error on group :"+g+" with agent "+groupMember);
//	e.printStackTrace();
//	}
//	} catch (/*CGRNotFound*/Exception e) {
//	System.err.println("!!!!!! This should not happen : kernel internal error on group :"+g);
//	e.printStackTrace();
//	}
//	}
//	return isEmpty();
//	}

}
