/*
* AgentAddress.java - Kernel: the kernel of MadKit
* Copyright (C) 1998-2008 Olivier Gutknecht, Fabien Michel, Jacques Ferber
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 2.1 of the License, or (at your option) any later version.

* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.

* You should have received a copy of the GNU Lesser General Public
* License along with this library; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
package madkit.kernel;



import java.util.NoSuchElementException;
import java.util.StringTokenizer;

/**
 * This class describe the unique identifiers used in the MadKit
 * kernel.  An identifier created with this class is guaranteed to be
 * unique.
 * 
 * This class has been designed to be extended and subclassed in the
 * future to handle various address type (FIPA, CORBA, MAF, JNA,
 * ...). The syntax in this actual class is MadKit-specific.
 * 
 * @author Olivier Gutknecht
 * @author Fabien Michel
 * @author Saber Mansour
 * @version 1.3 - 23/03/2007
 */

@SuppressWarnings("serial")
public class AgentAddress extends Object implements java.io.Serializable
{

	/** The Constant MADKIT_PROTOCOL. */
	//final private static String MADKIT_PROTOCOL = "mka";
	
	/** The launch nb. */
	private static int launchNb = 0;
	
	/** The protocol. */
	//private String protocol;
	
	/** The name. */
	private String name;
	
	/** The local id. */
	private int localID;
	
	/** The kernel. */
	private KernelAddress kernel=null;

  /**
   * Define a new AgentAddress.
   * 
   * @param n agent name
   * @param k local kernel address
   */

public AgentAddress(final String n, final KernelAddress k)
{
	launchNb++;
	//protocol = MADKIT_PROTOCOL;
	name     = n;
	localID = launchNb;
	kernel = k;
}

/**
 * Instantiates a new agent address.
 * 
 * @param anAddress the address
 */
public AgentAddress(final AgentAddress anAddress) // does somebody use that one ??!! and why ?
{
    localID =  anAddress.getLocalID();
	kernel = anAddress.getKernel();
	//	protocol = anAddress.getProtocol();
	name     = anAddress.getName();
}

/**
 * Attempts to define an AgentAddress from a string representation (MadKit syntax).
 * The official format is "mka:a,b@k" ("a" an agent name, "b" agent ID, "k" a kernel address)
 * 
 * @param def the def
 * 
 * @throws InvalidAddressException if the string cannot be parsed
 * 
 * public AgentAddress(String def1, String def2)
 * {
 * String def = def1 + def2;
 * try
 * {
 * int index = def.indexOf(":");
 * protocol = def.substring(0, index);
 * 
 * if ( (!(protocol.equals(MADKIT_PROTOCOL))) && (!(protocol.equals("request"))))
 * throw new NoSuchElementException("invalid protocol:"+protocol);
 * def = def.substring(index+1);
 * index = def.indexOf(",");
 * name=def.substring(0, index);
 * def = def.substring(index+1);
 * index = def.indexOf("@");
 * localID=Integer.parseInt(def.substring(0, index));
 * kernel=new KernelAddress(def.substring(index+1));
 * 
 * }
 * catch(final Exception e)
 * {
 * 
 * }
 * }
 */

/** Attempts to define an AgentAddress from a string representation (MadKit syntax).
    The official format is "mka:a,b@k" ("a" an agent name, "b" agent ID, "k" a kernel address)
    @param def string representation or agent name
    @throws InvalidAddressException if the string cannot be parsed */

public AgentAddress(String def) throws InvalidAddressException // TODO ask Jacques about that one and jess
{

	StringTokenizer tk=new StringTokenizer(def,":,@",true);
	try
	{
//		protocol =  tk.nextToken();
//		if ( (!(protocol.equals(MADKIT_PROTOCOL))) && (!(protocol.equals("request"))))
//			throw new NoSuchElementException("invalid protocol:"+protocol);
		tk.nextToken();
		name=tk.nextToken();
		tk.nextToken();
		localID=Integer.parseInt(tk.nextToken());
		tk.nextToken();
		StringBuffer sb = new StringBuffer();
		while(tk.hasMoreTokens())
			sb.append(tk.nextToken());
		kernel=new KernelAddress(sb.toString());
	}
	catch(NoSuchElementException e)
	{
		throw new InvalidAddressException("Invalid AgentAddress !"+def);
	}
}



/**
 * Returns a string representation of the AgentAddress (with MadKit syntax).
 * 
 * @return the string
 */
@Override
public String toString()
{
	return name + "," + localID + kernel;//.shortString();
}


/**
 * Check is this address is a local address.
 * 
 * @return true, if checks if is local
 */
public boolean isLocal()
{
	return kernel.isLocal();
}



/**
 * Compare two instances of AgentAddress.
 * 
 * @param o other agent address (can be void)
 * 
 * @return true, if equals
 */
@Override
public boolean equals(final Object o)
{
	if(! Kernel.onLineMode)
		return localID == ((AgentAddress)o).localID;
	try {
		final AgentAddress a = (AgentAddress)o;
		return (localID == a.localID && kernel.equals(a.getKernel()))    ;// && protocol.equals(a.protocol); agents do not have different protocol till now
	}
	catch (NullPointerException e) {
	}
	catch (ClassCastException e) {
	}
	return false;
}


/**
 * Returns this address protocol.
 * 
 * @return the protocol
 */
//public String getProtocol()    { return protocol; }

/**
 * Returns current agent name.
 * 
 * @return the name
 */
public String getName()    { return name; }

/**
 * Set current agent name. This does not affect uniqueness of the AgentAddress
 * 
 * @param n the n
 */
void setName(final String n) { name = n; }

/**
 * The discriminating part of an AgentAddress.
 * 
 * @return the local id
 */
public int getLocalID()  { return localID; }

/**
 * returns the address of the kernel on which the agent is currently running.
 * 
 * @return the kernel address
 */
public KernelAddress getKernel() { return kernel; }

/* (non-Javadoc)
 * @see java.lang.Object#hashCode()
 */
@Override
final public int hashCode(){
	return localID;
}


/**
 * Update.
 * 
 * @param ka the kernel address
 */
void update(final KernelAddress ka){
    launchNb++;
    localID = launchNb;
    kernel   = ka;
}

/* Saber Modification
  This method will be used by the original Kernel of the mobile agent, (the birth Agency), when this  one  
  returns home so that we will be sure that he will always has the same adress

*/

//FM : We have to check that, modifying the launchNb is wired
/**
 * Update.
 * 
 * @param add the add
 */
void update(final AgentAddress add)
{
    //launchNb = Integer.parseInt(add.getLocalID());
    localID =  add.getLocalID();
//		protocol = new String(add.getProtocol());
		name     = new String (add.getName());
		kernel   = add.getKernel();
}

/**
 * @return the launchNb
static synchronized final int getAgentCounter() {
	return launchNb;
}
 */

}
