package org.math.plot;

import java.awt.*;
import java.awt.image.*;
import java.io.*;
import java.util.*;

import javax.imageio.*;
import javax.swing.*;

import org.math.io.files.*;
import org.math.plot.canvas.*;
import org.math.plot.components.*;
import org.math.plot.plotObjects.*;
import org.math.plot.plots.*;

/**
 * BSD License
 * 
 * @author Yann RICHET
 */

public abstract class PlotPanel extends JPanel {

	private static final long serialVersionUID = 1L;

	public PlotToolBar plotToolBar;

	public PlotCanvas plotCanvas;

	public LegendPanel plotLegend;

	public final static String EAST = BorderLayout.EAST;

	public final static String SOUTH = BorderLayout.SOUTH;

	public final static String NORTH = BorderLayout.NORTH;

	public final static String WEST = BorderLayout.WEST;

	public final static String INVISIBLE = "INVISIBLE";

	public final static String SCATTER = "SCATTER";

	public final static String LINE = "LINE";

	public final static String BAR = "BAR";

	public final static String HISTOGRAM = "HISTOGRAM";

	public final static String BOX = "BOX";

	public final static String STAIRCASE = "STAIRCASE";

	public final static String GRID = "GRID";

	public final static Color[] COLORLIST = { Color.BLUE, Color.RED, Color.GREEN, Color.YELLOW, Color.ORANGE, Color.PINK, Color.CYAN, Color.MAGENTA };

	public PlotPanel(PlotCanvas _canvas, String legendOrientation) {
		plotCanvas = _canvas;
		setBackground(Color.WHITE);
		setLayout(new BorderLayout());

		addPlotToolBar(NORTH);

		addLegend(legendOrientation);

		add(plotCanvas, BorderLayout.CENTER);
	}

	public PlotPanel(PlotCanvas _canvas) {
		this(_canvas, INVISIBLE);
	}

	public void addLegend(String o) {
		if (o.equalsIgnoreCase(EAST)) {
			plotLegend = new LegendPanel(this, LegendPanel.VERTICAL);
			add(plotLegend, EAST);
		} else if (o.equalsIgnoreCase(SOUTH)) {
			plotLegend = new LegendPanel(this, LegendPanel.HORIZONTAL);
			add(plotLegend, SOUTH);
		} else if (o.equalsIgnoreCase(WEST)) {
			plotLegend = new LegendPanel(this, LegendPanel.VERTICAL);
			add(plotLegend, EAST);
		} else if (o.equalsIgnoreCase(NORTH)) {
			plotLegend = new LegendPanel(this, LegendPanel.HORIZONTAL);
			add(plotLegend, NORTH);
		} else if (o.equalsIgnoreCase(INVISIBLE)) {
			plotLegend = new LegendPanel(this, LegendPanel.INVISIBLE);
			// add(legends, BorderLayout.NORTH);
		} else
			System.err.println("Orientation " + o + " is unknonw.");
	}

	public void removeLegend() {
		remove(plotLegend);
	}

	public void setLegendOrientation(String o) {
		removeLegend();
		addLegend(o);
	}

	public void addPlotToolBar(String o) {
		if (o.equalsIgnoreCase(EAST)) {
			plotToolBar = new PlotToolBar(this);
			plotToolBar.setFloatable(false);
			add(plotToolBar, EAST);
		} else if (o.equalsIgnoreCase(SOUTH)) {
			plotToolBar = new PlotToolBar(this);
			plotToolBar.setFloatable(false);
			add(plotToolBar, SOUTH);
		} else if (o.equalsIgnoreCase(WEST)) {
			plotToolBar = new PlotToolBar(this);
			plotToolBar.setFloatable(false);
			add(plotToolBar, WEST);
		} else if (o.equalsIgnoreCase(NORTH)) {
			plotToolBar = new PlotToolBar(this);
			plotToolBar.setFloatable(false);
			add(plotToolBar, NORTH);
		} else
			System.err.println("Orientation " + o + " is unknonw.");
	}

	public void removePlotToolBar() {
		remove(plotToolBar);
	}

	public void setPlotToolBarOrientation(String o) {
		removePlotToolBar();
		addPlotToolBar(o);
	}

	// ///////////////////////////////////////////
	// ////// set actions ////////////////////////
	// ///////////////////////////////////////////

	public void setActionMode(int am) {
		plotCanvas.setActionMode(am);
	}

	public void setNoteCoords(boolean b) {
		plotCanvas.setNoteCoords(b);
	}

	public void setEditable(boolean b) {
		plotCanvas.setEditable(b);
	}

	public boolean getEditable() {
		return plotCanvas.getEditable();
	}

	public void setNotable(boolean b) {
		plotCanvas.setNotable(b);
	}

	public boolean getNotable() {
		return plotCanvas.getNotable();
	}

	// ///////////////////////////////////////////
	// ////// set/get elements ///////////////////
	// ///////////////////////////////////////////

	public Plot[] getPlots() {
		return plotCanvas.getPlots();
	}

	public Plot getPlot(int i) {
		return plotCanvas.getPlot(i);
	}

	public int getPlotIndex(Plot p) {
		return plotCanvas.getPlotIndex(p);
	}

	public Plotable[] getPlotables() {
		return plotCanvas.getPlotables();
	}

	public Plotable getPlotable(int i) {
		return plotCanvas.getPlotable(i);
	}

	public Axe getAxe(int i) {
		return plotCanvas.getGrid().getAxe(i);
	}

	public String[] getAxesScales() {
		return plotCanvas.getAxesScales();
	}

	//	 TODO axes labels are rested after addPlot... correct this.
	public void setAxesLabels(String... labels) {
		plotCanvas.setAxesLabels(labels);
	}

	public void setAxeLabel(int axe, String label) {
		plotCanvas.setAxeLabel(axe, label);
	}

	public void setAxesScales(String... scales) {
		plotCanvas.setAxesScales(scales);
	}

	public void setAxeScale(int axe, String scale) {
		plotCanvas.setAxeScale(axe, scale);
	}

	public void setFixedBounds(double[] min, double[] max) {
		plotCanvas.setFixedBounds(min, max);
	}

	public void setFixedBounds(int axe, double min, double max) {
		plotCanvas.setFixedBounds(axe, min, max);
	}

	public void includeInBounds(double... into) {
		plotCanvas.includeInBounds(into);
	}

	public void includeInBounds(Plot plot) {
		plotCanvas.includeInBounds(plot);
	}

	public void setAutoBounds() {
		plotCanvas.setAutoBounds();
	}

	public void setAutoBounds(int axe) {
		plotCanvas.setAutoBounds(axe);
	}

	// ///////////////////////////////////////////
	// ////// add/remove elements ////////////////
	// ///////////////////////////////////////////

	public void addLabel(String text, Color c, double... where) {
		plotCanvas.addLabel(text, c, where);
	}

	public void addBaseLabel(String text, Color c, double... where) {
		plotCanvas.addBaseLabel(text, c, where);
	}

	public void addPlotable(Plotable p) {
		plotCanvas.addPlotable(p);
	}

	public void removePlotable(Plotable p) {
		plotCanvas.removePlotable(p);
	}

	public void removePlotable(int i) {
		plotCanvas.removePlotable(i);
	}

	public int addPlot(Plot newPlot) {
		return plotCanvas.addPlot(newPlot);
	}

	protected Color getNewColor() {
		return COLORLIST[plotCanvas.plots.size() % COLORLIST.length];
	}

	public int addPlot(String type, String name, double[]... v) {
		return addPlot(type, name, getNewColor(), v);
	}

	public abstract int addPlot(String type, String name, Color c, double[]... v);

	public void setPlot(int I, Plot p) {
		plotCanvas.setPlot(I, p);
	}

	public void changePlotData(int I, double[]... XY) {
		plotCanvas.changePlotData(I, XY);
	}

	public void changePlotName(int I, String name) {
		plotCanvas.changePlotName(I, name);
	}

	public void changePlotColor(int I, Color c) {
		plotCanvas.changePlotColor(I, c);
	}

	public void removePlot(int I) {
		plotCanvas.removePlot(I);
	}

	public void removePlot(Plot p) {
		plotCanvas.removePlot(p);
	}

	public void removeAllPlots() {
		plotCanvas.removeAllPlots();
	}

	public void addQuantiletoPlot(int numPlot,boolean _symetric, double[]... q) {
		plotCanvas.addQuantiletoPlot(numPlot,_symetric, q);
	}

	public void addQuantiletoPlot(int numPlot, int numAxe,boolean _symetric, double... q) {
		plotCanvas.addQuantiletoPlot(numPlot, numAxe,_symetric, q);
	}

	public void addQuantiletoPlot(int numPlot, int numAxe, double q) {
		plotCanvas.addQuantiletoPlot(numPlot, numAxe, q);
	}

	public void toGraphicFile(File file) throws IOException {
		// otherwise toolbar appears
		plotToolBar.setVisible(false);

		Image image = createImage(getWidth(), getHeight());
		paint(image.getGraphics());
		image = new ImageIcon(image).getImage();

		BufferedImage bufferedImage = new BufferedImage(image.getWidth(null), image.getHeight(null), BufferedImage.TYPE_INT_RGB);
		Graphics g = bufferedImage.createGraphics();
		g.drawImage(image, 0, 0, Color.WHITE, null);
		g.dispose();

		// make it reappear
		plotToolBar.setVisible(true);

		try {
			ImageIO.write((RenderedImage) bufferedImage, "PNG", file);
		} catch (IllegalArgumentException ex) {
		}
	}

	public static void main(String[] args) {
		String man = "Usage: Plot <-2D|-3D, default = -2D> [options] <ascii file> [[options] other ascii file]\n[options] are:\n  -t <SCATTER|LINE|BAR>\n  -n name\n  -qX <1D ascii file>\n  -qY <1D ascii file>\n  -qZ <1D ascii file>\n";

		PlotPanel p = null;
		if (args[0].equals("-2D"))
			p = new Plot2DPanel();
		else if (args[0].equals("-3D"))
			p = new Plot3DPanel();
		else
			System.out.println(man);

		String leg = "INVISIBLE";
		String type = SCATTER;
		String name = "";
		double[][] v = null;
		LinkedList<double[]> qX = new LinkedList<double[]>();
		LinkedList<double[]> qY = new LinkedList<double[]>();
		LinkedList<double[]> qZ = new LinkedList<double[]>();

		for (int i = 1; i < args.length; i++) {
			if (args[i].equals("-l")) {
				leg = args[i + 1];
				i++;
			} else if (args[i].equals("-t")) {
				type = args[i + 1];
				i++;
			} else if (args[i].equals("-n")) {
				name = args[i + 1];
				i++;
			} else if (args[i].equals("-qX")) {
				//while (new File(args[i + 1]).exists()) {
				double[] q = ASCIIFile.readDouble1DArray(new File(args[i + 1]));
				qX.add(q);
				i++;
				//}
			} else if (args[i].equals("-qY")) {
				//while (new File(args[i + 1]).exists()) {
				double[] q = ASCIIFile.readDouble1DArray(new File(args[i + 1]));
				qY.add(q);
				i++;
				//}
			} else if (args[i].equals("-qZ")) {
				//while (new File(args[i + 1]).exists()) {
				double[] q = ASCIIFile.readDouble1DArray(new File(args[i + 1]));
				qZ.add(q);
				i++;
				//}
			} else {
				File input_file = new File(args[i]);
				if (input_file.exists()) {
					v = ASCIIFile.readDoubleArray(input_file);
					int n = p.addPlot(type, name, v);
					for (int j = 0; j < qX.size(); j++)
						p.addQuantiletoPlot(n, 0, false,qX.get(j));
					for (int j = 0; j < qY.size(); j++)
						p.addQuantiletoPlot(n, 1, false,qY.get(j));
					for (int j = 0; j < qZ.size(); j++)
						p.addQuantiletoPlot(n, 2, false,qZ.get(j));

					type = SCATTER;
					name = "";
					v = null;
					qX.clear();
					qY.clear();
					qZ.clear();
				} else {
					System.out.println("File " + args[i] + " unknown.");
					System.out.println(man);
				}
			}
		}
		p.setLegendOrientation(leg);
		FrameView f = new FrameView(p);
		f.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
	}

}
