/*
 * Genetic Package
 * Copyright (C) 2002-2007 Gregory Beurier
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package turtlekit2.genetic;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;
import java.util.Random;

import javax.swing.JPanel;

import turtlekit2.kernel.XMLAttributes;
/**
 * <p>Title : Pool  </p>
 * <p>Description : The class pool defines a population of genomes. A pool is a collection of homogeneous genomes used to achieve
 * the evolution/selection mechanism. </p>
 * <p></p>
 * <p>XML Attributes : given in the metamanger class</p>
 * <p>Name: the name of the pool. default is "GenomePool".</p>
 * <p>Population: the number of genomes in the pool and usually the number of agents. default is 10.</p>
 * <p>NbGenes: the number of creted Genes for this type in the pool. default is 1.</p>
 * <p></p>
 * <p></p>
 * @author Gregory Beurier
 */ 
public class Pool extends ArrayList<Genome> implements Serializable {
	final static int WAITING = -1;

	final static int ACTIVATED = 0;

	Random randomizer = new Random();

	int generation = 0;

	String poolName;

	String poolPath;

	static EvolutionManager myManager = new EvolutionManager();
	
	transient PoolGUI myGui = null;
	
	public Pool(String path, String name) {
		this.poolPath = path;
		this.poolName = name;
	}
	
	public Pool() {
		super();
	}

	public void createPool(Hashtable poolParameters, int populationSize) {
		try {
			System.out.println("Create " + poolParameters);
			if ((populationSize % 2) != 0 && populationSize != 1) {
				populationSize++;
			}
			for (int i = 0; i < populationSize; i++) {
				Genome newGenome = new Genome();
				newGenome.setPoolName(poolName);
				for (Iterator z = poolParameters.entrySet().iterator(); z
						.hasNext();) {
					Map.Entry e = (Map.Entry) z.next();
					newGenome.add((XMLAttributes) e.getValue());
				}
				newGenome.init();
				add(i, newGenome);
			}
			GeneticFileManager.savePool(poolPath+poolName,this);
		} catch (Exception e) {
			System.err.println("Instanciation problem!\n" + poolName + " \n"
					+ e);
		}
	}

	public Genome pop() {
		boolean allGenomeSimulated = true;
		if(size()!=1){
			for (int i = 0; i < size(); i++) {
				if (((Genome) get(i)).getState() == WAITING) {
					allGenomeSimulated = false;
					((Genome) get(i)).setState(ACTIVATED);
					return (Genome) get(i);
				}
			}
			if (allGenomeSimulated)
			{
	//			System.out.println("tout pop");
				newGeneration();
			}
		}else{return (Genome) get(0);}
		return pop();
	}

	public void resetState() {
		for (int i = 0; i < size(); i++) {
			((Genome) get(i)).setState(WAITING);
		}
	}

	public void addGeneToPool(Gene newGene){
		for (int i = 0; i < size(); i++) {
			((Genome) get(i)).add(newGene);
		}
	}
	/** *********************************************************************** */
	public void newGeneration() {
		myManager.reproduce(this);
		generation++;
		System.out.println("##### Generation: "
				+ new Integer(generation).toString() + " " + poolName
				+ " ##### "/* + deviation + " " + average */);
		refreshGui();
	}

	public void crossGenes1() {
		fitSortDesc();
		System.out.println(" ******* BEST GENOME ****** "
				+ ((Genome) get(0)).toString());
		for (int i = 0; i < (size() / 2); i++) {
			Genome father = (Genome) get(i);
			int motherIndex = randomizer.nextInt(size() / 2);
			Genome mother = (Genome) get(motherIndex);
			Genome child = (Genome) get(i + (size() / 2));

			// a dplacer dans les gnomes - a faire plus propre
			for (int l = 0; l < father.size(); l++) {
				Gene fatherGene = (Gene) father.get(l);
				Gene motherGene = (Gene) mother.get(l);
				if(fatherGene.isFixed()){
					child.set(l, (Gene) fatherGene.clone());
					((Gene)(child.get(l))).setFixed(true);
				}
				else ((Gene) child.get(l)).cross(fatherGene, motherGene);
			}
		}
	}
	
	public void crossGenes2() {
		fitSortDesc();
		System.out.println(" ******* BEST GENOME ****** " + ((Genome) get(0)).toString());
		for (int i = 0; i < (size()/2); i++) {
			Genome father = (Genome) get(i);
			int motherIndex = randomizer.nextInt(size() / 2);
			Genome mother = (Genome) get(motherIndex);
			Genome child = (Genome) get(i + (size() / 2));

			// a dplacer dans les gnomes - a faire plus propre
			for (int l = 0; l < father.size(); l++) {
				Gene fatherGene = (Gene) father.get(l);
				Gene motherGene = (Gene) mother.get(l);
				if(fatherGene.isFixed()){
					child.set(l, (Gene) fatherGene.clone());
					((Gene)(child.get(l))).setFixed(true);
				}
				else ((Gene) child.get(l)).cross(fatherGene, motherGene);
			}
		}
	}

	public void mutateGenes() {
		for (int i = 0; i < size(); i++) {
			Genome genome = (Genome) get(i);
			// a deplacer dans genome
			for (int l = 0; l < genome.size(); l++) {
				if(!((Gene) genome.get(l)).isFixed()) ((Gene) genome.get(l)).mutate();
			}
		}
	}

	public boolean add(Genome newGenome){
		boolean res=super.add(newGenome);
		refreshGui();
		return res;
	}
	/** *****************UTILS************************************************************************** */

	

	public void refreshGui(){
		if(myGui != null) myGui.refresh();
	}
	
	public void fitSortAsc() {
		for (int i = (size() - 1); i >= 0; i = i - 1) {
			int pos_max = i;
			for (int k = i - 1; k >= 0; k = k - 1)
				if (((Genome) get(k)).getFitness() > ((Genome) get(pos_max))
						.getFitness())
					pos_max = k;
			Genome tmpGenome = (Genome) get(pos_max);
			set(pos_max, get(i));
			set(i, tmpGenome);
		}
	}

	public void fitSortDesc() {
		for (int i = (size() - 1); i >= 0; i = i - 1) {
			int pos_max = i;
			for (int k = i - 1; k >= 0; k = k - 1)
				if (((Genome) get(k)).getFitness() < ((Genome) get(pos_max))
						.getFitness())
					pos_max = k;
			Genome tmpGenome = (Genome) get(pos_max);
			set(pos_max, get(i));
			set(i, tmpGenome);
		}
	}

	public String toString() {
		String result = "";
		for (int i = 0; i < size(); i++) {
			result += "\nGenome " + new Integer(i).toString() + " ";
			result += ((Genome) get(i)).toString();
		}
		return result;
	}

	public void setNewSize(int newSize){
		if(size()>newSize){
			int nbDestruction = size()-newSize;
			fitSortAsc();
			for(int i=0;i < nbDestruction;i++){
				remove(0);
			}
		}else {
			int nbCreation = newSize-size();
			for(int i=0;i < nbCreation ;i++) {
				createNewGenome();
			}
		}
		refreshGui();
	}
	
	public void createNewGenome(){
		if(size()>0){
			Genome newGenome = (Genome)((Genome)get(0)).clone();
			newGenome.reset();
			add(newGenome);
			refreshGui();
		}
	}
	/**
	 * ************************Getters &
	 * Setters*************************************************************
	 */

	public void reset(){
		for (Genome g : this) {
			g.reset();
		}
		refreshGui();
	}
	
	public void clear(){
		super.clear();
		refreshGui();
	}
	
	public Genome getGenome(int i) {
		return get(i);
	}

	public int getGeneration() {
		return generation;
	}

	public String getPoolName() {
		return poolName;
	}

	public int getSize() {
		return size();
	}

	public void setPoolName(String poolName) {
		this.poolName = poolName;
	}
	
	
	/******************************GRAPHICS****************************/
	
	
	public JPanel getPanel() {
		if(myGui == null) myGui = new PoolGUI(this);
		return myGui;
	}
}