/*
* TurtleEnvironment.java -TurtleKit - A 'star logo' in MadKit
* Copyright (C) 2000-2007 Fabien Michel
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/
package turtlekit.kernel;

import java.awt.Color;
import java.util.*;

import turtlekit.kernel.Turtle;

import madkit.kernel.AbstractAgent;
import madkit.kernel.ReferenceableAgent;

/**This Agent is the one who creates turtles,patches and who cares about managing them all (creation, death...)

  @author Fabien MICHEL
  @version 3.1 09/10/2006  */

public class TurtleEnvironment extends AbstractAgent implements ReferenceableAgent
{
    /**
	 * 
	 */
	private static final long serialVersionUID = -6870421308063427839L;
	Map<String, Integer> variables=null;
    Map<String, Double> diffuseCoef=null;
    Map<String, Double> evapCoef=null;
    final public Patch grid[][];
    final int x, y;
    int turtlesCount=-1;
    String simulationGroup;
    boolean wrap=true;
    ArrayList<Turtle> theTurtles = new ArrayList<Turtle>(5000);

    public TurtleEnvironment (final int width, final int height, final String group)
    {
	x = width;
	y = height;
 	grid = new Patch[x][y];
	initGrid();
 	this.simulationGroup = group;
    }

    public void displayOff(){
		for (int i=x-1; i >=0; i--)
			for (int j=y-1; j >=0; j--)
				if (grid[i][j].change) grid[i][j].change=false;
	}

    public void displayOn(){
		for (int i=x-1; i >=0; i--)
			for (int j=y-1; j >=0; j--)
		grid[i][j].change=true;}

    final void addVariables(final String[] var,final double[] values)
    {
	variables = new HashMap<String, Integer>(var.length);
	for(int i=0;i<var.length;i++) variables.put(var[i],new Integer(i));
	for (int i=x-1; i >=0; i--)
		for (int j=y-1; j >=0; j--)
		{
		    final double[] val = new double[values.length];
		    for(int k=0;k<var.length;k++) val[k]=values[k];
		    grid[i][j].variableValue = val;
		}
    }

    final void diffuseVariables(final String[] var,final double[] dc)
    {
	diffuseCoef = new HashMap<String, Double>(var.length);
	for(int i=0;i<var.length;i++)
	    diffuseCoef.put(var[i],new Double(dc[i]));
    }

    final void evapVariables(final String[] var,final double[] dc)
    {
	evapCoef = new HashMap<String, Double>(var.length);
	for(int i=0;i<var.length;i++)
	    evapCoef.put(var[i],new Double(dc[i]));
    }

    final void initNeighborhood()
    {
	Patch acc[];
	if (wrap)
		for (int i=x-1; i >=0; i--)
			for (int j=y-1; j >=0; j--)
		    {
			acc = new Patch[8];
			acc[0]=grid[(i+1) % x][j];
			acc[7]=grid[(i+1) % x][(j-1+y) % y];
			acc[6]=grid[i][(j-1+y) % y];
			acc[5]=grid[(i-1+x) % x][(j-1+y) % y];
			acc[4]=grid[(i-1+x) % x][j];
			acc[3]=grid[(i-1+x) % x][(j+1) % y];
			acc[2]=grid[i][(j+1) % y];
			acc[1]=grid[(i+1) % x][(j+1) % y];
			grid[i][j].setNeighborhood(acc);
		    }
	else
	    {
		for (int i=x-1; i >=0; i--)
			for (int j=y-1; j >=0; j--)
			{
			    if (i==0)
				if (j> 0 && j< y-1)
				    {
					acc = new Patch[5];
					acc[0]=grid[i+1][j];
					acc[1]=grid[i+1][j+1];
					acc[2]=grid[i][j+1];
					acc[3]=grid[i][j-1];
					acc[4]=grid[i+1][j-1];
				    }
				else
				    {
					if (j==0)
					    {
						acc= new Patch[3];
						acc[0]=grid[i+1][j];
						acc[1]=grid[i+1][j+1];
						acc[2]=grid[i][j+1];
					    }
					else
					    {
						acc= new Patch[3];
						acc[0]=grid[i+1][j];
						acc[1]=grid[i][j-1];
						acc[2]=grid[i+1][j-1];
					    }
				    }
			    else 
				if (i==(x-1))
				    if (j> 0 && j< y-1)
					{
					    acc = new Patch[5];
					    acc[0]=grid[i][j+1];
					    acc[1]=grid[i-1][j+1];
					    acc[2]=grid[i-1][j];
					    acc[3]=grid[i-1][j-1];
					    acc[4]=grid[i][j-1];
					}
				    else
					{
					    if (j==0)
						{
						    acc= new Patch[3];
						    acc[0]=grid[i][j+1];
						    acc[1]=grid[i-1][j+1];
						    acc[2]=grid[i-1][j];
						}
					    else
						{
						    acc= new Patch[3];
						    acc[0]=grid[i-1][j];
						    acc[1]=grid[i-1][j-1];
						    acc[2]=grid[i][j-1];
						}
					}
				else
				    if (j==0)
					{
					    acc = new Patch[5];
					    acc[0]=grid[i+1][j];
					    acc[1]=grid[i+1][j+1];
					    acc[2]=grid[i][j+1];
					    acc[3]=grid[i-1][j+1];
					    acc[4]=grid[i-1][j];
					}
				    else
					if (j==y-1)
					    {
						acc = new Patch[5];
						acc[0]=grid[i+1][j];
						acc[1]=grid[i-1][j];
						acc[2]=grid[i-1][j-1];
						acc[3]=grid[i][j-1];
						acc[4]=grid[i+1][j-1];
					    }
					else
					    {
						acc = new Patch[8];
						acc[0]=grid[i+1][j];
						acc[7]=grid[i+1][j-1];
						acc[6]=grid[i][j-1];
						acc[5]=grid[i-1][j-1];
						acc[4]=grid[i-1][j];
						acc[3]=grid[i-1][j+1];
						acc[2]=grid[i][j+1];
						acc[1]=grid[i+1][j+1];
					    }
			    grid[i][j].setNeighborhood(acc);
			}
	    }
    }
    
    final void initGrid()
    {
		for (int i=x-1; i >=0; i--)
			for (int j=y-1; j >=0; j--)
		    grid[i][j] = new Patch(this);
    }
    
    
final public void diffusion()
{
	if (diffuseCoef != null)
	{
		for(final Map.Entry<String, Double> e : diffuseCoef.entrySet())
		{
			final double coef = e.getValue();
			/*if(coef < 0.001)
				continue;*/
			final int index = variables.get(e.getKey());
			if (wrap)
			{
				final double give = coef/8;
				for (int i=x-1; i >=0; i--)
					for (int j=y-1; j >=0; j--)
					{
						grid[i][j].diffusion=grid[i][j].variableValue[index]*give;
						grid[i][j].variableValue[index]-=grid[i][j].variableValue[index]*coef;
					}
			}
			else
				for (int i=x-1; i >=0; i--)
					for (int j=y-1; j >=0; j--)
					{
						grid[i][j].diffusion=grid[i][j].variableValue[index]*(coef/grid[i][j].neighbors.length);
						grid[i][j].variableValue[index]-=grid[i][j].variableValue[index]*coef;
					}
			for (int i=x-1; i >=0; i--)
				for (int j=y-1; j >=0; j--)
					grid[i][j].update(index);
		}
	}
}
		
		
final public void evaporation()
{
	if (evapCoef != null)
	{
		for(final Map.Entry<String, Double> e : evapCoef.entrySet())
		{
			final double coef = e.getValue(); 
			/*if(coef < 0.001)
				continue;*/
			final int index = variables.get(e.getKey());
			for (int i=x-1; i >=0; i--)
				for (int j=y-1; j >=0; j--){
					grid[i][j].variableValue[index]-=grid[i][j].variableValue[index]*coef;
				}
		}
	}
}

    final synchronized void finalReset()
    {
    	for(Turtle t : theTurtles){
    		(t.position).removeAgent(t);
    		killAgent(t);
    		t=null;
    	}
    	theTurtles.clear();
    	turtlesCount=0;
    }

    final synchronized int createAgent(final Turtle agt, final int a, final int b)
    {
	turtlesCount++;
	agt.initialisation(a,b,this,turtlesCount,grid[a][b]); 
	grid[a][b].addAgent(agt);
	theTurtles.add(agt);
	launchAgent(agt,Turtle.TURTLE_DEFAULT_ROLE+turtlesCount,false);
	return turtlesCount;
    }
    final int addAgent(final Turtle agt){return createAgent(agt, (int) (Math.random() * x), (int) (Math.random() * y) );}
    final int addAgent(final Turtle agt, final int u, final int t) {	return createAgent(agt, u%x, t%y);  }

    final synchronized void removeTurtle(Turtle t)
    {
	(t.position).removeAgent(t);
	theTurtles.remove(t);
	if(t != null)
		killAgent(t);
	t=null;
    }

final Turtle[] turtlesAt(final int u,final int z)
{
	return grid[u][z].getTurtles();
}

final int turtlesCountAt(final int u, final int v){return grid[u][v].size();}

    final Color getPatchColor(final int u,final int v){    return (grid[u][v]).color;}
    final void setPatchColor(final Color c,final int u,final int v){(grid[u][v]).setColor(c);}

    final void moveTurtle(final int a,final int b,final Turtle t)
    {
	//final int u = (int) Math.round(a)%x;
	//final int v = (int) Math.round(b)%y;
    	Patch p = grid[a][b]; 
	if (p != t.position)
	    {
		(t.position).removeAgent(t);
		p.addAgent(t);
	    }
    }

    final public void activate()
    {
	requestRole(Launcher.COMMUNITY,simulationGroup,"world",null);
    }

	/**
	 * @param id
	 * @return the turtle with the id, null if not alive
	 */
	public Turtle getTurtleWithID(final int id) {
		for (Turtle t : theTurtles) {
			if(t.mySelf()==id)
				return t;
		}
		return null;
	}
}
