/*
* SiteAgent.java - Kernel: the kernel of MadKit
* Copyright (C) 1998-2008 Olivier Gutknecht, Fabien Michel, Jacques Ferber
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 2.1 of the License, or (at your option) any later version.

* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.

* You should have received a copy of the GNU Lesser General Public
* License along with this library; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
package madkit.kernel;

import static madkit.kernel.Kernel.COMMUNITIES;

import java.awt.AWTEvent;
import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.StringTokenizer;

import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.table.AbstractTableModel;

import madkit.boot.Madkit;

/** The SiteAgent represents the Kernel for communities and organizations synchronization management...
 @author Fabien Michel
  @version 1.0
  @since MadKit 3.0
 */

@SuppressWarnings("serial")
final class SiteAgent extends Agent
{
    public static final String SITE="site".intern();

	private AgentAddress myCommunicator=null;
	private final Map<String,Organization> organizations;
	private final Organization communities;
	private final KernelAgent kernelAgent;
	private final Collection<AgentAddress> distantKernels;
	private siteAgentGUI gui;

SiteAgent(final Map<String,Organization> organizations, KernelAgent agt)
{
	this.organizations = organizations;
	communities = new Organization();
	organizations.put(COMMUNITIES,communities);
	kernelAgent = new KernelAgent();
	distantKernels = new HashSet<AgentAddress>();
}

////////////////////////////////////////////////////	LIFE CYCLE
@Override
public void initGUI()
{
	gui = new siteAgentGUI(this);
	setGUIObject(gui);
}

@Override
final public void activate()
{
	setDebug(Madkit.debug);
    setName("SITE:"+this.getAddress().getKernel().getHost());
	createCommunity(Kernel.DEFAULT_COMMUNITY);
	//createGroup(true,"communities","public",null,null);
	//requestRole("communities","public","site",null);
	createGroup(false,"communications",null,null);
	requestRole("communications","site",null);
	//createGroup(false,"local",null,null);
	createGroup(false,"system",null,null);
    requestRole("system",SITE,null);
	kernelAgent.setLocalOrg(organizations.get(Kernel.DEFAULT_COMMUNITY));
	try {
		getCurrentKernel().launchAgent(kernelAgent,"KernelAgent",getCurrentKernel(),false);
	} catch (LaunchAgentException e) {
		e.printStackTrace();
	}
}

@Override
final public void live()
{   try{
	while(true)
	{
		final Message m = waitNextMessage();
		if(m instanceof SynchroMessage)
			handleSynchroMessage((SynchroMessage) m);
		else if(m instanceof ConnectionMessage)
			handleConnectionMessage((ConnectionMessage) m);
		else if(m instanceof NetworkRequest)
			handleNetworkRequest((NetworkRequest) m);
		else if(m instanceof StringMessage){
                    handleMessage((StringMessage) m);
		} else
			debug("receive an unknown message type :"+m);
	}
      } catch(final Exception ex){
          debug("Error : "+ex);
          ex.printStackTrace();
      }
}

protected void handleMessage(final StringMessage msg){
       // System.out.println("::: handling message :"+msg);
        final String content = msg.getString();
        final StringTokenizer st = new StringTokenizer(content," ");
        String token = st.nextToken();
        if (token.equalsIgnoreCase("$url")){
            token = st.nextToken();
            AgentAddress ag = getAgentWithRole("system","browser");
            if (ag == null){
                final AbstractAgent a = AbstractMadkitBooter.getBooter().makeJavaAgent(this,"agents.system.WebBrowserAgent",true);
                ag = a.getAddress();
                if (ag == null){
                  System.err.println("Error: cannot create WebBrowserAgent upon reception of a $url request");
                }
            }
            sendMessage(ag,new StringMessage("$goto "+token));
        } else if (token.equalsIgnoreCase("$message")){
          try {
            final String s = content.substring("$message".length()+1,content.length());
            AgentAddress ag = getAgentWithRole("system","pager");
            if (ag == null){
                final AbstractAgent a = AbstractMadkitBooter.getBooter().makeJavaAgent(this,"madkit.system.Pager",true);
                ag = a.getAddress();
                if (ag == null){
                  System.err.println("Error: cannot create a Pager upon reception of a $message request");
                }
            }
            sendMessage(ag,new StringMessage("$display "+"Received a remote message from "
                        +msg.getSender()+": \n"+s));
           } catch(final Exception e){}
        }
}

@Override
public void end()
{
	System.err.println("SITE AGENT KILLED !!!!!!!!!!!!!");
	System.err.println("UNSTABLE SYSTEM ....");
}

///////////////////////////////////////////////////////// Handling messages

final void handleSynchroMessage(final SynchroMessage m)
{
	//debug("handling synchro message"+m);
	final String communityName = m.community;
	if(connectedWith(communityName))
	{
		Organization localOrg = organizations.get(communityName);
		if(localOrg == null)
		{
			kernelAgent.callHooks(Kernel.NEW_COMMUNITY,communityName);
			localOrg = new Organization();
			organizations.put(communityName,localOrg);
		}
		switch(m.code)
		{
			case Kernel.CREATE_GROUP:
				localOrg.createGroup(m.initiator, true, m.groupName, m.newGroup.getDescription(), m.newGroup.getGroupIdentifier());
				kernelAgent.callHooks(Kernel.CREATE_GROUP,  new AGRTrio(m.initiator,communityName,m.groupName,null));
				break;
			case Kernel.ADD_MEMBER_ROLE:
				try {
					localOrg.requestRole(m.initiator, m.groupName, m.roleName, m.memberCard);
				} catch (final CGRException e) {
					if(Madkit.debug){
						debug("problem handling synchro message\n"+m);
						debug(e.toString());
						e.printStackTrace();
					}
				}
				kernelAgent.callHooks(Kernel.ADD_MEMBER_ROLE,  new AGRTrio(m.initiator,communityName,m.groupName,m.roleName));
				break;
			case Kernel.LEAVE_GROUP:
				try {
					if(localOrg.leaveGroup(m.initiator, m.groupName))
						removeCommunity(communityName);
				} catch (final CGRException e1) {
					if(Madkit.debug){
						debug("problem handling synchro message\n"+m);
						debug(e1.toString());
						e1.printStackTrace();
					}
				}
				kernelAgent.callHooks(Kernel.LEAVE_GROUP,  new AGRTrio(m.initiator,communityName,m.groupName,null));
				break;
			case Kernel.NEW_COMMUNITY:
				establishConnectionWith(m.getSender(), true);
				break;
			case Kernel.DELETE_COMMUNITY:
				localOrg.removeAgentsFromKernel(m.getSender().getKernel());
				if(localOrg.isEmpty())
				{
					removeCommunity(communityName);
					kernelAgent.callHooks(Kernel.DELETE_COMMUNITY,communityName);
				}
				break;
			case Kernel.REMOVE_MEMBER_ROLE:
				try {
					if(localOrg.leaveRole(m.initiator, m.groupName, m.roleName))
					{
						removeCommunity(communityName);
					}
					kernelAgent.callHooks(Kernel.REMOVE_MEMBER_ROLE,  new AGRTrio(m.initiator,communityName,m.groupName,m.roleName));
				} catch (final CGRException e) {
					if(Madkit.debug){
						debug("problem handling synchro message\n"+m);
						e.setCommunity(communityName);
						debug(e.toString());
						e.printStackTrace();
					}
				}
				break;
			case Kernel.MIGRATION:
				if(Madkit.debug)
					debug("receiving a migration "+m.ref.toString());
				getCurrentKernel().receiveAgent(m.ref);
				break;
		}
	}
	else {
		switch(m.code)
		{
			case Kernel.NEW_COMMUNITY:
				kernelAgent.callHooks(Kernel.NEW_COMMUNITY,communityName);
				if (Madkit.debug)
					debug("new community detected");
				break;
			case Kernel.DELETE_COMMUNITY:
				kernelAgent.callHooks(Kernel.DELETE_COMMUNITY,communityName);
				if (Madkit.debug)
					debug("a community has been deleted ");//+organizations);
				if (gui != null)
                                  gui.refreshCommunities();//establishConnectionWith(m.getSender(), true);
				break;
		}
		if (gui != null)
	        gui.refreshCommunities();//establishConnectionWith(m.getSender(), true);
	}
}

final void handleConnectionMessage(final ConnectionMessage m)
{
	if(m.isTheFirstMessage())
	{
		//debug("handling connection request\n"+m);
		getCurrentKernel().synchronizeKernel( m.getOrgs(), true);
		establishConnectionWith(m.getSender(),false);
	}
	else{
		//debug("receiving ack for connection\n"+m);
		getCurrentKernel().synchronizeKernel( m.getOrgs(), false);
	}
	final AgentAddress sender = m.getSender();
	if(distantKernels.add(sender)){
		if (Madkit.debug)
			debug("*********************Connecting to "+sender+"*******************");
		kernelAgent.callHooks(Kernel.CONNECTED_TO, sender.getKernel());
		kernelAgent.callHooks(Kernel.ADD_MEMBER_ROLE, sender, COMMUNITIES, Kernel.DEFAULT_COMMUNITY, SITE);
        }
}

synchronized final void handleNetworkRequest(final NetworkRequest m)
{
	switch(m.getRequestCode())
	{
		case NetworkRequest.INJECT_MESSAGE:
			try
			{
				final Message theMessage = (Message) m.getArgument();
				getCurrentKernel().sendLocalMessage(theMessage);
				if(! (theMessage instanceof PrivateMessage))
					if(theMessage instanceof SecuredMessage)
						kernelAgent.callHooks(Kernel.SEND_MESSAGE, theMessage.clone());
					else
						kernelAgent.callHooks(Kernel.SEND_MESSAGE, theMessage);
			}
			catch (final MessageException me) {if (getDebug()) System.err.println(me);}
			break;
		case NetworkRequest.CONNECTION_REQUEST:
			if(m.getSender().equals(myCommunicator))
			{
		  		establishConnectionWith((AgentAddress) m.getArgument(),true);
		  		//distantKernels.add(m.getArgument());
		  	}
		  	break;
		case NetworkRequest.DECONNECTED_FROM:
	  		distantKernels.remove(m.getArgument());
		  	deconnection((String) m.getArgument());
		  	break;
		case NetworkRequest.BE_COMMUNICATOR:
			if(myCommunicator == null)
			{
				redisplayMyGUI();
				myCommunicator = m.getSender();
			}
			break;
		case NetworkRequest.STOP_COMMUNICATOR:
			if(m.getSender().equals(myCommunicator))
			{
				disposeMyGUI();
				myCommunicator = null;
				for(final Iterator<Map.Entry<String, Organization>> i = organizations.entrySet().iterator();i.hasNext();)	{
					final Map.Entry<String, Organization> e = i.next();
					if(e.getValue().removeAgentsFromKernel(null)) {
						leaveGroup("communities", e.getKey());
						i.remove();
					}
				}
				distantKernels.clear();
				System.gc();
				System.runFinalization();
			}
			break;
		case NetworkRequest.GET_AVAILABLE_DESTINATIONS:
			sendMessage(m.getSender(), new NetworkRequest(NetworkRequest.GET_AVAILABLE_DESTINATIONS, distantKernels.toArray(new KernelAddress[distantKernels.size()])));
			break;
		case NetworkRequest.REQUEST_MIGRATION:
			if (Madkit.debug)
				println("receiving a migration request of "+m.getSender());
			// kernelAgent.callsHook(Kernel.MIGRATION...	à rajouter
			tryMigration((KernelAddress) m.getArgument(),m.getSender());
			break;
	}
}


////////////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////	CONNECTION

//////////////////////////////////////////////////////////////////////////////////////////
final synchronized void establishConnectionWith(final AgentAddress distantKernel, final boolean first)
{
	final Map<String, Organization> orgs = new HashMap<String, Organization>();
	
	for (final Map.Entry<String, Organization> entry : organizations.entrySet()) {
		if(connectedWith(entry.getKey())){
			orgs.put(entry.getKey(), entry.getValue().exportOrg());
		}
	}
	sendMessage(distantKernel, new ConnectionMessage(orgs,first));
	if (Madkit.debug)
		debug("sending connection message to "+distantKernel);
}

final synchronized void deconnection(final String id)
{
	for(Iterator<AgentAddress> i = distantKernels.iterator();i.hasNext();)
	{
		final AgentAddress distantK = i.next();
		if(distantK.getKernel().getID().equals(id))
		{
			if (Madkit.debug)
				debug("disconnected from "+id);
			for(final Iterator<Map.Entry<String, Organization>> j = organizations.entrySet().iterator();j.hasNext();)
			{
				final Map.Entry<String, Organization> e = j.next();
				if(e.getValue().removeAgentsFromKernel(distantK.getKernel()))
				{
					final String theCommunity = e.getKey();
					leaveGroup("communities", theCommunity);
					sendAll(new SynchroMessage(Kernel.LEAVE_GROUP,getAddress(), COMMUNITIES, theCommunity,null, null));
					sendAll(new SynchroMessage(Kernel.DELETE_COMMUNITY, theCommunity));
					j.remove();
				}
			}
			kernelAgent.callHooks(Kernel.DISCONNECTED_FROM, distantK.getKernel());
			kernelAgent.callHooks(Kernel.REMOVE_MEMBER_ROLE, distantK, COMMUNITIES, Kernel.DEFAULT_COMMUNITY, SITE);
			i.remove();
			break;
		}
	}
	if (gui != null)
          gui.refreshCommunities();
}

////////////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////	SYNCHRONIZATION

//////////////////////////////////////////////////////////////////////////////////////////
final synchronized void updateDistantOrgs(final AgentAddress initiator, final String community, final String groupName)
{
	Collection<AgentAddress> receivers = null;
	try {
		receivers = communities.getRolePlayers(community,SITE);
	} catch (final CGRException e) {
		e.setCommunity(community);
		if(! community.equals(COMMUNITIES)){
		}
	}
	if(receivers != null){
		final Organization orgOfTheGroup = organizations.get(community);
		for (final AgentAddress distantSite : receivers)
		{
			if(! distantSite.equals(getAddress()))
			{
				final Message m = new SynchroMessage(initiator, community, orgOfTheGroup.get(groupName),groupName);
				m.setSender(getAddress());
				m.setReceiver(distantSite);
				sendDistantMessage(m);
			}
		}
	}
}

final synchronized void updateDistantOrgs(final int code,final AgentAddress initiator, final String community, final String groupName,final String roleName, final Object memberCard) throws CGRException
{

	final Message m = new SynchroMessage(code,initiator, community, groupName,roleName, memberCard);
	for (final AgentAddress receiver : communities.getRolePlayers(community,SITE)){
		if(! receiver.equals(getAddress())) {
			m.setSender(getAddress());
			m.setReceiver(receiver);
			sendDistantMessage(m);
		}
	}
//	
//	Collection<AgentAddress> receivers = null;
//	try {
//		receivers = communities.getRolePlayers(community,SITE);
//	} catch (final CGRException e) {
//		e.setCommunity(community);
//		if(! community.equals(COMMUNITIES)){
//			System.err.println("!!!!!! This should not happen : kernel internal error ");
//			e.printStackTrace();
//		}
//	}
//	if(receivers != null) {
//		for (final AgentAddress receiver : receivers)	{
//			if(! receiver.equals(getAddress())) {
//				final Message m = new SynchroMessage(code,initiator, community, groupName,roleName, memberCard);
//				m.setSender(getAddress());
//				m.setReceiver(receiver);
//				sendDistantMessage(m);
//			}
//		}
//	}
}

final synchronized void updateDistantOrgs(final SynchroMessage m) 
{
	String community = m.community;
	if(connectedWith(community))
		try{
			if(community.equals(COMMUNITIES))
				community=m.groupName;
			for (final AgentAddress receiver : communities.getRolePlayers(community,SITE)){
				if(! receiver.equals(getAddress())) {
					m.setSender(getAddress());
					m.setReceiver(receiver);
					sendDistantMessage(m);
				}
			}
		}
		catch (CGRException e) {
//			e.setCommunity(m.community);
//				println("!!!!!! This should not happen : kernel internal error ");
//				e.printStackTrace();
		}
}


@SuppressWarnings("deprecation")
final synchronized void tryMigration(final KernelAddress destination, final AgentAddress traveler)
{
	final AbstractAgent ref = Kernel.getReference(traveler);
	final Message message = new SynchroMessage(ref);
	
  for(final Iterator<AgentAddress> i = distantKernels.iterator();i.hasNext();)
	{
		final AgentAddress potentialReceiver = i.next();
		if(potentialReceiver.getKernel().equals(destination))
		{
			message.setReceiver(potentialReceiver);
			break;
		}
	}

	if(message.getReceiver() != null && ref != null)
	{
		//kernelAgent.callshook 	...  updateDistantOrgs(Kernel.MIGRATION,traveler,null,null,null);

		if(ref instanceof Mobile)
		{
			((Agent)ref).getAgentThread().stop();
			if(Kernel.getAddress().equals(((Mobile)ref).getMyAgency())) // agentMobile natif
			{
				final AgentAddress birthAddress =((Mobile)ref).getMyBirthAddress();// A remplacer  en appelant la m�thode en utilisant localAgents
				final Mirror mirror=	((Mobile)ref).createMirror(destination);
			
				getCurrentKernel().removeAgentFromOrganizations(traveler);
				getCurrentKernel().removeReferenceOf(traveler);// It is important to put the remove operation
				ref.setCurrentKernel(null);
				// launch the mirrorAgent.
				//mirror.launch(currentKernel,birthAddress,"mirroAgentOf"+getName());
				currentKernel.launchMirror(mirror,birthAddress,"mirroAgentOf"+getName(),getCurrentKernel(),true);
				((Mobile)ref).createMirror(destination);
			}
			else
			{
				getCurrentKernel().removeAgentFromOrganizations(traveler);
				getCurrentKernel().removeReferenceOf(traveler);// It is important to put the remove operation
				ref.setCurrentKernel(null);
			}
			//ref.messageBox=null;
		message.setSender(getAddress());
		sendDistantMessage(message);
		}

	// SABER_Modification
	// Here we must get all the distributed groups where the mobile agent is registred
	// Of course he will leave these groups but we will send a request to register him again But
	// this time his adress will be the Birth Address
	//
	//
	}
}


////////////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////	DISTANT MESSAGING MANAGEMENT

//////////////////////////////////////////////////////////////////////////////////////////

// TODO to check : myCommunicator has to be a trusted agent or cloning m is required

/**
 * Send a distant message by delegating to the agent that plays the role of Communicator inside the Organization.
 * 
 * @param message the message to send to a distant kernel
 * 
 * @return true, if successful
 */
final synchronized void sendDistantMessage(final Message message)
{
	if(myCommunicator != null) {
		Message m2 = null;
//		if(message instanceof SecuredMessage){
//			m2 = new KernelMessage(KernelMessage.NO_REQUEST,Kernel.SEND_MESSAGE,message.clone());
//		}
//		else
		m2 = new KernelMessage(KernelMessage.NO_REQUEST,Kernel.SEND_MESSAGE,message);
		m2.setSender(getAddress());
   		m2.setReceiver(myCommunicator);
   		try	{
			getCurrentKernel().sendLocalMessage(m2);
			kernelAgent.callHooks(Kernel.SEND_MESSAGE, (message instanceof SecuredMessage) ? message.clone() : message);
		}
		catch(final MessageException mexc) {
			if (Madkit.debug) {
				println("Unable to send distant message !!");
				mexc.printStackTrace();
			}
		}
	}
}

private void sendAll(final Message m) {
	m.setSender(getAddress());
	for(final AgentAddress distantK : distantKernels)	{
		m.setReceiver(distantK);
		sendDistantMessage(m);
	}
}
////////////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////	COMMUNITY MANAGEMENT

//////////////////////////////////////////////////////////////////////////////////////////
synchronized void joinCommunity(final String communityName)
{
	if(! connectedWith(communityName))
	{
		if (Madkit.debug)
			debug("connecting to community : "+communityName);
		if(createGroup(true,COMMUNITIES,communityName,null,null) == 1)
			sendAll(new SynchroMessage(getAddress(), COMMUNITIES, communities.get(communityName),communityName));
//			if(getDebug()){
//				debug(e.toString());
//				e.printStackTrace();
//			}
		requestRole(COMMUNITIES,communityName,SITE,null);		//must be optimized
		sendAll(new SynchroMessage(Kernel.ADD_MEMBER_ROLE,getAddress(), COMMUNITIES, communityName,SITE, null));
		sendAll(new SynchroMessage(Kernel.NEW_COMMUNITY,communityName));
	}
	else{
		if (Madkit.debug)
			debug("already connected with "+communityName);
	}
}
synchronized void leaveCommunity(final String communityName)
{
	if (Madkit.debug)
		debug("deconnecting community "+communityName);
	//final Organization organization = (Organization) organizations.get(communityName);
	if(organizations.get(communityName).removeAgentsFromKernel(null))
	{
		leaveGroup(COMMUNITIES,communityName);
		sendAll(new SynchroMessage(Kernel.LEAVE_GROUP, getAddress(), COMMUNITIES, communityName, null, null));
		organizations.remove(communityName);
		kernelAgent.callHooks(Kernel.DELETE_COMMUNITY,communityName);
	}
	else
	{
		leaveRole(COMMUNITIES,communityName,SITE);
		sendAll(new SynchroMessage(Kernel.REMOVE_MEMBER_ROLE,getAddress(), COMMUNITIES, communityName, "site", null));
		sendAll(new SynchroMessage(Kernel.DELETE_COMMUNITY, communityName));
	}
	if (gui != null) 
		gui.refreshCommunities();
}

synchronized void removeCommunity(final String communityName)
{
	if (Madkit.debug)
		debug("removing community : " + communityName);
	organizations.remove(communityName);
	kernelAgent.callHooks(Kernel.DELETE_COMMUNITY,communityName);
	leaveGroup(COMMUNITIES,communityName);
	sendAll(new SynchroMessage(Kernel.LEAVE_GROUP,getAddress(), COMMUNITIES, communityName,null, null));
	sendAll(new SynchroMessage(Kernel.DELETE_COMMUNITY, communityName));
	if (gui != null)
          gui.refreshCommunities();
}

synchronized boolean connectedWith(final String community)
{
	if(community.equals(COMMUNITIES) || communities.isPlayingRole(getAddress(),community,SITE))
		return true;
	return false;
}

synchronized Organization createCommunity(final String communityName)
{
	if(! organizations.containsKey(communityName))
	{
		if (Madkit.debug)
			debug("creating community : "+communityName);
		final Organization org = new Organization();
		organizations.put(communityName.intern(),org);
		createGroup(true,COMMUNITIES,communityName,null,null);	
		requestRole(COMMUNITIES,communityName,SITE,null);
		sendAll(new SynchroMessage(getAddress(), COMMUNITIES, communities.get(communityName),communityName));
		sendAll(new SynchroMessage(Kernel.ADD_MEMBER_ROLE,getAddress(), COMMUNITIES, communityName,SITE, null));
		sendAll(new SynchroMessage(Kernel.NEW_COMMUNITY,communityName));
		kernelAgent.callHooks(Kernel.NEW_COMMUNITY,communityName);
		if(hasGUI())
			gui.refreshCommunities();
		if (Madkit.debug)
			debug("community <"+communityName+"> created");
		return org;
	}
	return null;
}

synchronized void refreshCommunities() {
	if (gui != null)
          gui.refreshCommunities();
}

synchronized String[] getCommunities() {
	final Collection<String> c = new HashSet<String>();
	for (final String string : communities.getGroups()) {
		try {
			if(communities.isPlayingRole(getAddress(),string,Group.MEMBER_DEFAULT_ROLE) || communities.get(string).getRolePlayers(SITE).size() > 0 )
				c.add(string);
		} catch (final CGRException e) {
//			e.setCommunity(COMMUNITIES);
//			System.err.println("!!!!!! This should not happen : kernel internal error ");
//			e.printStackTrace();
		}
	}
	return c.toArray(new String[c.size()]);
}

// do not kill the agent on window closing
    @Override
	public void windowClosing(final AWTEvent we){
    }

    @Override
	public int createGroup(final boolean distributed, final String communityName, final String groupName, String description, final GroupIdentifier theIdentifier){
		if (description == null)
			description = groupName;
		try {
			currentKernel.createGroup(getAddress(), distributed, communityName, groupName, description, theIdentifier);
		} catch (OrganizationRequestException e) {
			if(getDebug()){
				debug(e.getClass().getSimpleName()+e.getMessage());
				e.printStackTrace();
			}
			return e.getCode();
		}
		return 1;
	}


}

////////////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////	PRIVATE MESSAGE CLASSES

//////////////////////////////////////////////////////////////////////////////////////////

@SuppressWarnings("serial")
final class ConnectionMessage extends Message  implements PrivateMessage
{
	private Map<String, Organization> org;
	private boolean first;

	ConnectionMessage(final Map<String, Organization> org, final boolean first)
	{
		this.org = org;
		this.first=first;
	}
	final Map<String, Organization> getOrgs(){return org;}
	final boolean isTheFirstMessage(){return first;}

}


@SuppressWarnings("serial")
final class SynchroMessage extends Message  implements PrivateMessage
{
	int code;
	AgentAddress initiator=null;
	String groupName=null,roleName=null, community=null;
	Group newGroup=null;
	Object memberCard=null;
	AbstractAgent ref=null;

	SynchroMessage(AgentAddress initiator, String community, Group theNewGroup,String groupName)
	{
		code = Kernel.CREATE_GROUP;
		newGroup = theNewGroup;
		this.community = community;
		this.groupName = groupName;
		this.initiator = initiator;
	}

	SynchroMessage(AbstractAgent initiator)
	{
		this.ref = initiator;
		this.community = Kernel.DEFAULT_COMMUNITY;
		code = Kernel.MIGRATION;
	}

	SynchroMessage(int code, AgentAddress initiator, String community, String groupName,String roleName, Object memberCard)
	{
		this.code = code;
		this.groupName = groupName;
		this.community = community;
		this.roleName = roleName;
		this.initiator = initiator;
		this.memberCard = memberCard;
	}

	SynchroMessage(int code, String community)
	{
		this.community = community;
		this.code = code;
	}

	int getCode(){return code;}
	
	@Override
	public String toString() {
		return super.toString()+"<"+ community+","+groupName+","+roleName+"> \nwith code : "+code+"\non agent "+initiator;
	}
}




////////////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////	GUI

//////////////////////////////////////////////////////////////////////////////////////////

@SuppressWarnings("serial")
final class siteAgentGUI extends JPanel implements ItemListener
{
	SiteAgent myAgent;

	siteAgentGUI(SiteAgent agt){
		setSize(200,300);
		myAgent = agt;
		//setTitle(myAgent.getName());
		refreshCommunities();
	}



	void refreshCommunities(){
		removeAll();
		setLayout(new BorderLayout());
		String[] communityNames = myAgent.getCommunities();
		Object[][] data=new Object[communityNames.length][2];
		for(int i = 0; i < communityNames.length; i++)
		{
			data[i][0]=communityNames[i];
			data[i][1]=new Boolean(myAgent.connectedWith(communityNames[i]));
			//JCheckBox c = new JCheckBox(communityNames[i],myAgent.connectedWith(communityNames[i]));
			//c.addItemListener(this);
			//tmp.add(c);
		}
		//JPanel tmp = new JPanel();
		//tmp.setLayout(new GridLayout(communityNames.length,1,5,5));
		//tmp.setBorder(BorderFactory.createEmptyBorder(20,20,20,20));

		SitesTableModel sitesModel = new SitesTableModel(this,data);
        JTable table = new JTable(sitesModel);
        table.setPreferredScrollableViewportSize(new Dimension(200, 100));

        //Create the scroll pane and add the table to it.
        JScrollPane scrollPane = new JScrollPane(table);

        //Add the scroll pane to this window.
        add(scrollPane, BorderLayout.CENTER);

		add("North",new JLabel("Available communities"));
		//add("Center",tmp);
		validate();
	}

    void communityChange(boolean b, String name){
        if (b)
            myAgent.joinCommunity(name);
        else
            myAgent.leaveCommunity(name);
    }

	public void itemStateChanged(ItemEvent e){
		Object source = e.getItemSelectable();
		if (e.getStateChange() == ItemEvent.DESELECTED)
		{
			myAgent.leaveCommunity( ((JCheckBox)source).getText());
			}
			else
		{
			myAgent.joinCommunity( ((JCheckBox)source).getText());
			}

	}

}

@SuppressWarnings("serial")
class SitesTableModel extends AbstractTableModel {
        final String[] columnNames = {"Sites",
                                      "connected"};
        Object[][] data;
        siteAgentGUI gui;

		SitesTableModel(siteAgentGUI g,Object[][] obj){
			data = obj;
            gui = g;
	}
	
        public int getColumnCount() {
            return columnNames.length;
        }

        public int getRowCount() {
            return data.length;
        }

        @Override
		public String getColumnName(int col) {
            return columnNames[col];
        }

        public Object getValueAt(int row, int col) {
            return data[row][col];
        }

        @Override
		public Class<? extends Object> getColumnClass(int c) {
            return getValueAt(0, c).getClass();
        }

        @Override
		public boolean isCellEditable(int row, int col) {
            if ((data[row][0].equals(Kernel.DEFAULT_COMMUNITY)) || (col < 1)) {
                return false;
            } else {
                return true;
            }
        }

        @Override
		public void setValueAt(Object value, int row, int col) {
            data[row][col] = value;
            gui.communityChange(((Boolean)value).booleanValue(),(String)data[row][0]);
            fireTableCellUpdated(row, col);
        }
    }
