/*
 * KernelAddress.java - Kernel: the kernel of MadKit
 * Copyright (C) 1998-2008 Olivier Gutknecht, Fabien Michel, Jacques Ferber
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package madkit.kernel;

import java.net.InetAddress;
import java.util.Date;
import java.util.NoSuchElementException;
import java.util.StringTokenizer;

/** This class identifies a Kernel Address. It is the naming of the
    "home agent platform" in a transport address or communication
    scheme (as proposed in FIPA-1999-17). The syntax is now "host:id"
    but will probably change to "mka://host/id" to map better to the
    FIPA specification

  @author Olivier Gutknecht
  @version d6 - 09/01/97 */

@SuppressWarnings("serial")
public class KernelAddress extends Object implements java.io.Serializable
{
	private String host;
	private String ID = "0";
	//Added by saber
	private String kernelName="DefaultMadkitName";
	private boolean supportMobility = false;
	// end of saber contribution

	/**
	 * Instantiates a new kernel address for the offlineMode 
	 */
	KernelAddress(){
		host = null;
		ID = null;
		kernelName = null;
	}
	
	/** Defines a new unique KernelAddress on the local host 
	@param ipnumber define is the host must be in numeric form or resolved though the DNS
	 */
	public KernelAddress(boolean ipnumber)
	{
		ID = buildID();
		try
		{
			if (ipnumber)
				host = InetAddress.getLocalHost().getHostAddress();
			else
				host = InetAddress.getLocalHost().getHostName();
		} 
		catch(Exception e) 
		{ 
			host = "localKernel";
		}
	}

	public KernelAddress(boolean ipnumber, String ipaddress)
	{
		ID = buildID();
		try
		{
			String hostname = InetAddress.getLocalHost().getHostName();
			InetAddress addresses[] = InetAddress.getAllByName(hostname);
			for(int i=0; i<addresses.length; i++) 
			{
				if (ipaddress.compareTo(addresses[i].getHostAddress()) == 0)
				{
					host = ipaddress;
					return;
				}
			}
			if (ipnumber)
				host = InetAddress.getLocalHost().getHostAddress();
			else
				host = hostname;
		} 
		catch(Exception e) 
		{ 
			host = "localKernel";
		}
	}

	/** Tries to parse an KernelAddress from a string representation. 
    The official format is "a:b" (with a an Internet host and an b unique ID )

    @param def string representation of Kernel address
    @throws InvalidAddressException if the string cannot be parsed in
    the KernelAddress String format */

	public KernelAddress(String def) throws InvalidAddressException 
	{
		StringTokenizer tk=new StringTokenizer(def,":",false);
		try 
		{
			if (tk.countTokens() == 2)
			{
				host=tk.nextToken();
				ID = tk.nextToken();
			}
//			Added by saber
			else 	if (tk.countTokens() == 4)
			{
				host=tk.nextToken();
				ID = tk.nextToken();
				kernelName = tk.nextToken();
				if(tk.nextToken()=="1")
					supportMobility = true;				
				else
					supportMobility = false; // default false is the status of Mobility
			}
//			End of Saber contribution
			else
				throw 
				new InvalidAddressException("Invalid count on KernelAddress:"+
						def+tk.countTokens());
		}
		catch(NoSuchElementException e)
		{
			throw 
			new InvalidAddressException("Invalid element on KernelAddress !"+ 
					def+tk);
		}
	}

	/**
	 * Check is this address is a local address.
	 * 
	 * @return true, if the address belongs to this kernel
	 */
	public boolean isLocal()
	{
		return (this.equals(Kernel.getAddress()));
	}

	/**
	 * Compares two KernelAdresses.
	 * 
	 * @param o other address (can be void)
	 * 
	 * @return true, if equals
	 */
	@Override
	public boolean equals(Object o)
	{
		if (o instanceof KernelAddress)
			return (toString().equals(o.toString()));
		return false;
	}
	/** Return the host name encoded in this KernelAddress */
	public String getHost()
	{ return host; }
	/** Return the anonymous ID associated */
	public String getID() 
	{ return ID; }
	
	private String buildID(){
		//return "K"+ (new Date()).getTime();
		return Long.toString(new Date().getTime()).substring(9);
	}

	/** Returns the canonical string representation for this platform address */
	@Override
	public String toString()
	{
		return ("@"+host+":"+ID);
	}
	// Added by Saber in place of toString() because somewhere in connection process .toString() is used 
	// To compare two adresses and i don't like to modify all this process for this reason
	public String getInformation()
	{
		if(!supportMobility)
			return (host+":"+ID);
		else
			return("Agency Name="+kernelName+"\t:"+host+":"+ID);
	}

//	public String shortString()
//	{
//		return ("@"+host+":"+ID);
//		//return host+":"+ID.substring(10);
//	}
	// added by saber

	public void setKernelName(String name)
	{
		kernelName = name;
	}
	public String getKernelName()
	{
		return kernelName;
	}

	public void enableMobility()
	{
		supportMobility=true;
	}

	//@TODO big security issue here : contact Saber 
	public void disableMobility()
	{
		supportMobility=false;
	}

	public boolean supportMobility()
	{
		return supportMobility;
	}
	// end of saber contribution
}

final class LocalKernelAddress extends KernelAddress{

	@Override
	public boolean equals(Object o) {
		return true; // :)
	}
	
	@Override
	public boolean isLocal() {
		return true; // :)
	}
	@Override
	public String toString() {
		return "";
	}
}