/*
* TurtleKit - A 'reactive simulation platform' using MadKit Kernel
* Copyright (C) 2000-2007 Fabien Michel, Gregory Beurier
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

package turtlekit2.tools;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Container;
import java.awt.FlowLayout;
import java.awt.Graphics;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.awt.event.MouseWheelEvent;
import java.awt.event.MouseWheelListener;

import java.util.ArrayList;

import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.SwingConstants;

import turtlekit2.kernel.*;
/**
 * <p>Titre : AdvancedViewer</p>
 * <p>Description : AdvancedViewer is a simulation world displayer agent. It displays flavors/pheromones according to RGB canals and
 * permits flavors observations and modifications in the environment in real time. </p>
 * <p>Use : To display a flavor on a RGB canal, click on the color then on the flavor name (mixing canals is allowed). To drop a flavor, click on drop then on the flavor name, then click within the environment. 
 * To modify the displaying scale, click on a color then roll mouse wheel over the environment. To display a flavor/pheromone quantity, do as drop
 * and move the mouse cursor over the environment.</p>
 * <p></p>
 * <p>XML Attributes : None</p>
 * @author Gregory Beurier
 */


public class AdvancedViewer extends Viewer implements MouseListener, MouseMotionListener,ActionListener, MouseWheelListener{
	XMLAttributes attrib = new XMLAttributes();
	int patchIndex = 0;
	final static int RED = 0;
	final static int GREEN = 1;
	final static int BLUE = 2;
	final static int DROP = 3;
	final static int BLACK = 4;
	
	int modifiedColor = RED;
	
	double redFactor = 255;
	double blueFactor = 255;
	double greenFactor = 255;
	double blackFactor = 255;
	
	double maxColoredRed = 100000000;
	double maxColoredGreen = 100000000;
	double maxColoredBlue = 100000000;
	double maxColoredBlack = 100000000;
	
	double quantityRed = 0;
	double quantityGreen = 0;
	double quantityBlue = 0;
	double quantityBlack = 0;
	
	double maxLocalColoredRed = -10000;
	double maxLocalColoredGreen = -10000;
	double maxLocalColoredBlue = -10000;
	double maxLocalColoredBlack = -10000;
	
	ArrayList<String> redColors = new ArrayList<String>();
	ArrayList<String> greenColors = new ArrayList<String>();
	ArrayList<String> blueColors = new ArrayList<String>();
	ArrayList<String> blackColors = new ArrayList<String>();
	
	String dropPher = "";
	
	JMenuBar menuBar;
    JMenu pheroMenu;
    JMenuItem menuItem;
    JButtonColor redb, greenb, blueb, blackb, drop, pheroButtons, turtleb;
	JFrame f = new JFrame("Custom Viewer COnfiguration");
	JLabel cord, factors, phero;
	boolean turtleDisplay = true;
	
	/**No usage*/
	public void init() {}
	
	/**MadKit usage, no redefinition*/
	public void initGUI() {
		super.initGUI();
		createFrame();
		getOnScreen().addMouseWheelListener(this);
		getOnScreen().addMouseListener(this);
		getOnScreen().addMouseMotionListener(this);
	}

	/**Graphical usage, no redefinition*/
	public void createFrame(){

	    f.setSize(400, 250);
	    Container content = f.getContentPane();
	    content.setLayout(new FlowLayout());
	    turtleb = new JButtonColor("Display Turtles");
	    redb = new JButtonColor("Red");
	    redb.setBackground(Color.red);
	    redb.setBorderPainted(false);
	    greenb = new JButtonColor("Green");
	    greenb.setBackground(Color.green);
	    greenb.setBorderPainted(false);
	    blueb = new JButtonColor("Blue");
	    blueb.setBackground(Color.blue);
	    blueb.setBorderPainted(false);
	    blackb = new JButtonColor("Black");
	    blackb.setBackground(Color.black);
	    blackb.setBorderPainted(false);
	    drop = new JButtonColor("Drop");
	    drop.setBackground(Color.white);
	    drop.setBorderPainted(false);
	    content.add(turtleb);
	    content.add(redb);
	    content.add(greenb);
	    content.add(blueb);
	    content.add(blackb);
	    content.add(drop);
        cord = new JLabel("Coord X - Y", SwingConstants.CENTER);
        factors = new JLabel("Factors", SwingConstants.CENTER);
        phero = new JLabel("PheroQtty", SwingConstants.CENTER);
        JPanel panel = new JPanel();
	    f.getContentPane().add(panel, BorderLayout.CENTER);
	    panel.add(cord);
	    panel.add(factors);
	    panel.add(phero);
	    turtleb.addActionListener(this);
	    redb.addActionListener(this);
	    greenb.addActionListener(this);
	    blueb.addActionListener(this);
	    blackb.addActionListener(this);
	    drop.addActionListener(this);
	    menuBar = new JMenuBar();
	    pheroMenu = new JMenu("Pheromones");
	    pheroMenu.setMnemonic(KeyEvent.VK_R);
	    menuBar.add(pheroMenu);
	    pheroMenu.addActionListener(this);
	    menuItem = new JMenuItem("Null");
	    pheroMenu.add(menuItem);
	    menuItem.addActionListener(this);
	    for(int i=0; i<getFlavors().size();i++){
	    	//menuItem = new JMenuItem(getFlavorName(i));
	    	pheroButtons = new JButtonColor(getFlavorName(i));
	    	pheroMenu.add(menuItem);
	    	content.add(pheroButtons);
	    	menuItem.addActionListener(this);
	    	pheroButtons.addActionListener(this);
	    }
	    f.setJMenuBar(menuBar);
	    f.setVisible(true);
	}
	
	/**Standard graphical representation of turtle. Can be disabled/enabled in real time. 
	 * You can override this method in order to give a special graphic representation of your turtles.*/
	public void paintTurtle(Graphics g, Turtle t,int x,int y,int cellS)
    {
		if(turtleDisplay){
			g.setColor(t.getColor());
			g.fillRect(x,y,cellS,cellS);
		}
	}
	
	/**The Paintpatch method has been overriden in order to achieve specific representation of flavors. 
	 * Each flavor can be represented as a function of the Red/Blue/Green/Black canals.*/
	public void paintPatch(Graphics g, Patch p,int x,int y,int CellSize)
	{
		quantityRed = 0;
		quantityGreen = 0;
		quantityBlue = 0;
		quantityBlack = 0;
		try{
			if(redColors.size()>0){
				for(int i =0; i< redColors.size();i++){
					quantityRed += ((double)p.getVariableValue(redColors.get(i)));
				}
				if(quantityRed > maxLocalColoredRed) maxLocalColoredRed = quantityRed;
				quantityRed = (quantityRed/(maxColoredRed+1))*redFactor;
				if(quantityRed>255) quantityRed=255;
				if(quantityRed<0) quantityRed=0;
			}else quantityRed = 0;
			
			if(greenColors.size()>0){
				for(int i =0; i< greenColors.size();i++){
					quantityGreen += ((double)p.getVariableValue(greenColors.get(i)));
				}
				if(quantityGreen > maxLocalColoredGreen) maxLocalColoredGreen = quantityGreen;
				quantityGreen = (quantityGreen/(maxColoredGreen+1))*greenFactor;
				if(quantityGreen>255) quantityGreen=255;
				if(quantityGreen<0) quantityGreen=0;
			}else quantityGreen = 0;
			
			if(blueColors.size()>0){
				for(int i =0; i< blueColors.size();i++){
					quantityBlue += ((double)p.getVariableValue(blueColors.get(i)));
				}
				if(quantityBlue > maxLocalColoredBlue) maxLocalColoredBlue = quantityBlue;	
				quantityBlue = (quantityBlue/(maxColoredBlue+1))*blueFactor;
				if(quantityBlue>255) quantityBlue=255;
				if(quantityBlue<0) quantityBlue=0;
			}else quantityBlue = 0;
			
			if(blackColors.size()>0){
				for(int i =0; i< blackColors.size();i++){
					quantityBlack += ((double)p.getVariableValue(blackColors.get(i)));
				}
				if(quantityBlack > maxLocalColoredBlack) maxLocalColoredBlack = quantityBlack;			
				quantityBlack = (quantityBlack/(maxColoredBlack+1))*blackFactor;
				if(quantityBlack>255) quantityBlack=255;
				if(quantityBlack<0) quantityBlack=0;
			}else quantityBlack = 0;

			
			quantityRed -= quantityBlack;
			quantityGreen -= quantityBlack;
			quantityBlue -= quantityBlack;
			if(quantityRed<0) quantityRed=0;
			if(quantityGreen<0) quantityGreen=0;
			if(quantityBlue<0) quantityBlue=0;
			
			
			g.setColor(new Color((int)quantityRed,(int)quantityGreen,(int)quantityBlue));
			g.fillRect(x,y,CellSize,CellSize);
			if(x == 0 && y ==0){
				maxColoredRed = maxLocalColoredRed;
				maxLocalColoredRed = -100000;
				maxColoredGreen = maxLocalColoredGreen;
				maxLocalColoredGreen = -100000;
				maxColoredBlue = maxLocalColoredBlue;
				maxLocalColoredBlue = -100000;
				maxColoredBlack = maxLocalColoredBlack;
				maxLocalColoredBlack = -100000;
				}
			
		}
		catch(Exception e){System.err.println("Error Viewer " + e);}
	}
	
	
	/**GUI method*/
	public void updateColorList(ArrayList<String> colorList, String pheromone){
		if(colorList.contains(pheromone)) colorList.remove(pheromone);
		else colorList.add(pheromone);
	}
	
	/**GUI method*/
    public void actionPerformed(ActionEvent e) { 
    	 Object s=e.getSource();
    	 if(s == turtleb){
    	 	turtleDisplay = !turtleDisplay;
    	 }
    	 else if(s == redb) {
    	 	modifiedColor = RED;
    	 }
    	 else if(s == greenb) {
    	 	modifiedColor = GREEN;
    	 }
    	 else if(s == blueb) {
    	 	modifiedColor = BLUE;
    	 }
    	 else if(s == blackb) {
     	 	modifiedColor = BLACK;
     	 }
    	 else if(s == drop) {
    	 	modifiedColor = DROP;
    	 }
    	 else if(flavorsContains(e.getActionCommand())){
		 	if(modifiedColor == RED){
		 		updateColorList(redColors,e.getActionCommand());
		 		((JButtonColor)s).switchColor(RED);
		 	}else if(modifiedColor == GREEN){
		 		updateColorList(greenColors,e.getActionCommand());
		 		((JButtonColor)s).switchColor(GREEN);
		 	}else if(modifiedColor == BLUE){
		 		updateColorList(blueColors,e.getActionCommand());
		 		((JButtonColor)s).switchColor(BLUE);
		 	}else if(modifiedColor == BLACK){
		 		updateColorList(blackColors,e.getActionCommand());
		 		((JButtonColor)s).switchColor(BLACK);
		 	}else if(modifiedColor == DROP){
		 		dropPher = e.getActionCommand();
		 		drop.setText("Actual Drop:" + dropPher);
		 	}
    	 }
    } 
    
    /**GUI method*/
	 public void mousePressed(MouseEvent e){}
	 /**GUI method*/
	 public void mouseReleased(MouseEvent e){}
	 /**GUI method*/
	 public void mouseEntered(MouseEvent e){}
	 /**GUI method*/
	 public void mouseExited(MouseEvent e){}
	 /**GUI method*/
	 public void mouseClicked(MouseEvent e){}
	 /**GUI method. It drops Pheromones/Flavors on the environment*/
	 public void mouseDragged(MouseEvent e){
		 int x = e.getX()/getCellSize();
	     int y = e.getY()/getCellSize();
	     if(flavorsContains(dropPher)) patchGrid[x][envHeight - y].incrementPatchVariable(dropPher,1000);
	 }
	 /**GUI method. When a Flavor is selected to be dropped, it displays its quantity in the environment.*/   
	 public void mouseMoved(MouseEvent e){
	    	cord.setText("X:" +e.getX()/getCellSize() + " Y:"+ (e.getY())/getCellSize());
	    	if(!dropPher.equals("") && !dropPher.equals(null)){
	    		phero.setText(new Double(patchGrid[e.getX()/getCellSize()][envHeight - e.getY()/getCellSize()].getVariableValue(dropPher)).toString());
	    	}
	    }
	 /**GUI method. WheelMouse modify the scale of flavors displaying as a function of RGB canals.*/
	  public void mouseWheelMoved(MouseWheelEvent e) {
	        String message;
	        int notches = e.getWheelRotation();
	        if(modifiedColor == RED){
	        	redFactor += notches*20;
		 	}else if(modifiedColor == GREEN){
		 		greenFactor += notches*20;
		 	}else if(modifiedColor == BLUE){
		 		blueFactor += notches*20;
		 	}else if(modifiedColor == BLACK){
		 		blackFactor += notches*20;
		 	}
	        factors.setText("R:" + redFactor + " G:"+ greenFactor + " B:" + blueFactor + " Bl:" + blackFactor);
	     }
}

/**GUI Class*/
class JButtonColor extends JButton{
	int rValue = 0;
	int gValue = 0;
	int bValue = 0;
	int blValue = 0;
	final static int RED = 0;
	final static int GREEN = 1;
	final static int BLUE = 2;
	final static int BLACK = 4;
	
	JButtonColor(String name){
		super(name);
	}
	
	public void switchColor(int color){
		if(color == RED){
			if(rValue >0) rValue = 0;
			else rValue = 255;
		}else if(color == BLUE){
			if(bValue > 0) bValue = 0;
			else bValue = 255;
		}else if(color == GREEN){
			if(gValue >0) gValue = 0;
			else gValue = 255;
		}else if(color == BLACK){
			if(blValue >0) blValue = 0;
			else blValue = 255;
		}
		
		if(rValue ==0 && gValue==0 && bValue ==0 && blValue == 0) setBackground(Color.lightGray);
		else setBackground(new Color(rValue, gValue, bValue));
	}
}

