/*
 * Genetic Package Copyright (C) 2002-2007 Gregory Beurier
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or any later version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package turtlekit2.genetic;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;
import java.util.Vector;

import madkit.boot.Madkit;

import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import turtlekit2.kernel.Flavor;
import turtlekit2.kernel.Launcher;
import turtlekit2.kernel.Observer;
import turtlekit2.kernel.Turtle;
import turtlekit2.kernel.TurtleProbe;
import turtlekit2.kernel.XMLAttributes;

/**
 * <p>Title : MetaManager Class.  </p>
 * <p>Description : MetaManager schedules fitness manager and allocates pools and genomes to agents. The metamanager
 * is launched by the launcher.</p>
 * <p></p>
 * @author Gregory Beurier
 */ 
public class MetaManager extends Observer {
	
	ArrayList<GTurtle> managedTurtles = new ArrayList<GTurtle>();

	Launcher launcher;

	MetaManagerGUI myGui;
	
	Hashtable<Manager, Integer> activationTable = new Hashtable<Manager, Integer>();

	Hashtable<Comparable, Genome> genomeAllocation = new Hashtable<Comparable, Genome>();

	Hashtable<String, Pool> poolTable = new Hashtable<String, Pool>();

	Vector flavors;

	ArrayList<Manager> managersList = new ArrayList<Manager>();

	NodeList poolsNodes, managersNodes;

	double stopWeight = 0;

	TurtleProbe probe;

	int simulationDuration = 0;
	
	int defaultRefresh = 0;

	String path = "";

	/** Constructor called by the launcher */
	public MetaManager(Launcher l, NodeList poolNodes, NodeList managersList, Vector flavors, int dRefresh) {
		launcher = l;
		this.flavors = flavors;
		initializeGenomePools(poolNodes);
		initAllManagers(managersList);
		defaultRefresh = dRefresh;
		createFrame();
		System.out.println("MetaManager Created");
	}

	public void setup() {}

	public void activate() {
		super.activate();
	}

	public void createFrame() {
		myGui = new MetaManagerGUI("Metamanager", poolTable, this);
		myGui.setVisible(true);
	}
	
	/** clear invoked when a new generation is requested */
	public void clear() {
		for (Iterator z = activationTable.entrySet().iterator(); z.hasNext();) {
			Map.Entry e = (Map.Entry) z.next();
			killAgent((Manager) (e.getValue()));
		}
	}

	
	/** the manage method allocates genomes to agents and defines the way pools are used */
	public void manage(GTurtle turtle) {
		/** ********Giving genome to turtle*************** */
		XMLAttributes agentAttributes = turtle.getAttributes();
		String agentPool = agentAttributes.getString("Pool");
		int allocationType = 0;
		if(agentAttributes.containsKey("GeneticMode")) allocationType = agentAttributes.getInt("GeneticMode");
		Genome g = null;
		if (allocationType == 0)
			g = poolTable.get(agentPool).pop(); // 1 genome per
		// agent
		else if (allocationType == 1) { // 1 genome per node
			if (genomeAllocation.containsKey(new Integer(agentAttributes.getInt("Node"))))
				g = genomeAllocation.get(new Integer(agentAttributes.getInt("Node")));
			else {
				g = poolTable.get(agentPool).pop();
				genomeAllocation.put(new Integer(agentAttributes.getInt("Node")), g);
			}
		} else if (allocationType == 2) { // 1 genome per role && per pool
			if (genomeAllocation.containsKey(agentAttributes.getString("Role"))) {
				g = genomeAllocation.get(agentAttributes.getString("Role"));
				if (g.getPoolName().equals(agentPool)) {
				} else {
					g = poolTable.get(agentPool).pop();
					genomeAllocation.put(agentAttributes.getString("Role"), g);
				}
			} else {
				g = poolTable.get(agentPool).pop();
				genomeAllocation.put(agentAttributes.getString("Role"), g);
			}
		} else if (allocationType == 3) { // 1 genome per pool
			if (genomeAllocation.containsKey(agentPool)) {
				g = genomeAllocation.get(agentPool);
			} else {
				g = poolTable.get(agentPool).pop();
				genomeAllocation.put(agentPool, g);
			}
		} else if (allocationType == 4) { // 1 genome per group && per pool
			if (genomeAllocation.containsKey(agentAttributes.getString("GroupNode"))) {
				g = genomeAllocation.get(agentAttributes.getString("GroupNode"));
				if (g.getPoolName().equals(agentPool)) {
				} else {
					g = poolTable.get(agentPool).pop();
					genomeAllocation.put(agentAttributes.getString("GroupNode"), g);
				}
			} else {
				g = poolTable.get(agentPool).pop();
				genomeAllocation.put(agentAttributes.getString("GroupNode"), g);
			}
		}
		System.out.println(g);
		turtle.setGenome((Genome) g.clone());

		/** ****Giving turtle to managers*********** */
		for (int i = 0; i < managersList.size(); i++) {
			Manager manager = managersList.get(i);
			String monitor = "all";
			if(manager.getAttrib().containsKey("Monitor")) monitor = manager.getAttrib().getString("Monitor");
			if (monitor.equals("all")) {
				manager.addAgentToMonitor(turtle, g);
			} else if (monitor.equals("pool")) {
				if (manager.getAttrib().getString("Monitored").equals(agentPool))
					manager.addAgentToMonitor(turtle, g);
			} else if (monitor.equals("role")) {
				if (manager.getAttrib().getString("Monitored").equals(
						agentAttributes.getString("Role")))
					manager.addAgentToMonitor(turtle, g);
			} else if (monitor.equals("node")) {
				if (manager.getAttrib().getString("Monitored").equals(
						agentAttributes.getString("Node")))
					manager.addAgentToMonitor(turtle, g);
			} else if (monitor.equals("group")) {
				if (manager.getAttrib().getString("Monitored").equals(
						agentAttributes.getString("GroupNode")))
					manager.addAgentToMonitor(turtle, g);
			}
		}

		managedTurtles.add(turtle);
	}

	protected void initializeGenomePools(NodeList poolsNodes) {
		try {
			if (poolsNodes.getLength() > 0) {
				for (int i = 0; i < poolsNodes.getLength(); i++) {
					Hashtable<Integer, XMLAttributes> poolParameters = new Hashtable<Integer, XMLAttributes>();
					Element pool = (Element) poolsNodes.item(i);
					String poolName = "GenomePool";
					if(pool.hasAttribute("Name")) poolName = pool.getAttribute("Name");
					int populationSize = 10; 
					if(pool.hasAttribute("Population")) populationSize = Integer.parseInt(pool.getAttribute("Population"));
					NodeList genes = pool.getElementsByTagName("Gene");
					String geneType = "turtlekit2.genetic.BitGene";
					for (int j = 0; j < genes.getLength(); j++) {
						Element geneDescription = (Element) genes.item(j);
						NamedNodeMap attributeList = geneDescription
								.getAttributes();
						int listLength = attributeList.getLength();
						if(geneDescription.hasAttribute("GeneClass")) geneType = geneDescription.getAttribute("GeneClass");
						int nbGenes = 1;
						if(geneDescription.hasAttribute("NbGenes")) nbGenes = Integer.parseInt(geneDescription.getAttribute("NbGenes"));
						XMLAttributes geneAttribute = new XMLAttributes();
						for (int k = 0; k < listLength; k++) {
							geneAttribute.put(attributeList.item(k)
									.getNodeName(), geneDescription
									.getAttribute(attributeList.item(k)
											.getNodeName()));
						}
						geneAttribute.put("nbFlavors", new Integer(flavors
								.size()).toString());
						poolParameters.put(new Integer(j), geneAttribute);
					}
					Pool newPool = new Pool(path, poolName);
					newPool.createPool(poolParameters, populationSize);
					poolTable.put(poolName, newPool);
				}
			}
		} catch (Exception e) {
			System.err.println("Genome initialisation problem !\n" + e);
		}
	}

	public void initAllManagers(NodeList managersList) {
		for (int i = 0; i < managersList.getLength(); i++) {
			Node managerList = managersList.item(i);
			NodeList managersNodes;
			managersNodes = ((Element) managerList).getElementsByTagName("Manager");
			initializeManagers(managersNodes);
		}
	}

	public void initializeManagers(NodeList managersNodes) {
		try {
			if (managersNodes.getLength() > 0) {
				for (int i = 0; i < managersNodes.getLength(); i++) {
					/** ***init patches*** */
					Element managerDescription = (Element) managersNodes
							.item(i);
					NamedNodeMap attributeList = managerDescription
							.getAttributes();
					int listLength = attributeList.getLength();
					XMLAttributes managerTable = new XMLAttributes();
					for (int k = 0; k < listLength; k++) {
						managerTable.put(attributeList.item(k).getNodeName(),
								managerDescription.getAttribute(attributeList.item(k).getNodeName()));
					}
					newManager(managerTable);
				}
			}
		} catch (Exception e) {
			System.err.println("Manager Initialisation problem !\n" + e);
		}
	}

	public void newManager(XMLAttributes managerParameters) {
		String managerType = (String) managerParameters.get("ManagerClass");
		try {
			Class managerClass = Madkit.getClassLoader().loadClass(managerType);
			Manager newManager = (Manager) (managerClass.newInstance());
			newManager.setManagerAttributes(managerParameters);
			newManager.setMetaManager(this);
			managersList.add(newManager); // A virer et ne garder que
			// l'activation table
			if (managerParameters.containsKey("Refresh"))
				activationTable.put(newManager, new Integer(managerParameters.getInt("Refresh")));
			else
				activationTable.put(newManager, new Integer(1));
			System.out.println("Launching " + managerParameters);
			launcher.addObserver(newManager, false);
		} catch (Exception e) {
			System.err.println("Loading manager " + managerType + "\n" + e);
		}
		System.out.println("Manager init:" + managerParameters);
	}

	synchronized public void watch() {
		simulationDuration++;

		for (Iterator z = activationTable.entrySet().iterator(); z.hasNext();) {
			Map.Entry e = (Map.Entry) z.next();
			if (simulationDuration % (((Integer) (e.getValue())).intValue()) == 0) {
				((Manager) (e.getKey())).update(simulationDuration);
			}
		}
		if(defaultRefresh != 0)
			if (simulationDuration % defaultRefresh == 0)
				askNewPopulation(1);
	}

	public void askNewPopulation(double stopPriority) {
		stopWeight += stopPriority;
		if (stopWeight >= 1) {
			managedTurtles.clear();
			for (int i = 0; i < managersList.size(); i++) {
				(managersList.get(i)).clear();
			}
			launcher.newPopulation();
			stopWeight = 0;
		}
	}

	public void clearSimulation() {
		for (int i = 0; i < envWidth; i++) {
			for (int j = 0; j < envHeight; j++) {
				if (patchGrid[i][j].getTurtles().length > 0) {
					Turtle t[] = patchGrid[i][j].getTurtles();
					for (int k = 0; k < t.length; k++) {
						t[k].die();
					}
				}
				for (int l = 0; l < flavors.size(); l++) {
					patchGrid[i][j].setPatchVariable(((Flavor) (flavors.get(l))).getName(), 0);
				}
			}
		}
	}

	public void setPoolsNodes(NodeList poolsNodes) {
		this.poolsNodes = poolsNodes;
	}

	public void setManagersNodes(NodeList managersNodes) {
		this.managersNodes = managersNodes;
	}

	public void setLaunchedTurtles(ArrayList<GTurtle> launchedTurtles) {
		this.managedTurtles = launchedTurtles;
	}

	public void setPath(String path) {
		this.path = path;
	}

	public void setDefaultRefresh(int defaultRefresh) {
		this.defaultRefresh = defaultRefresh;
	}
}
