/*
 * Genetic Package Copyright (C) 2002-2007 Gregory Beurier
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or any later version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package turtlekit2.genetic;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Hashtable;

import javax.swing.JLabel;
import javax.swing.JPanel;

import madkit.boot.Madkit;
import turtlekit2.kernel.XMLAttributes;

/**
 * <p>Title : Genome Class. Collection of Genes. </p>
 * <p>Description : In biology the genome of an organism is its whole hereditary information and is encoded in the DNA (or, for some viruses, RNA). 
 * This includes both the genes and the non-coding sequences of the DNA. The Genome class manage the collection of a turtles genes for 
 * selection, reproduction, mutation, getters and setters. A genome own a fitness value that represents the quality
 * of the genome for the task to achieve in simulation. A turtlekit genome is usually composed by homogeneous type of genes.
 * The genomes are managed by a Metamanager and can be saved..</p>
 * @author Gregory Beurier
 */  

public class Genome extends ArrayList<Gene> implements Serializable {
	final static int WAITING = -1;

	final static int ACTIVATED = 0;

	transient GenomeGUI myGui = null;

	transient JLabel fitnessLabel = null;

	int state = WAITING;

	transient Hashtable<Object,Double> managerTable = new Hashtable<Object,Double>();

	double fitness = 0;

	String poolName;

	public Genome() {
		fitness = 0;
	}

	/** add a gene (from this xml parameters) in the collection/genome */
	@SuppressWarnings("unchecked")
	public void add(XMLAttributes geneAttribute) {
		try {
			int nbThisGene = geneAttribute.getInt("NbGenes");
			System.out.println("Creating " + nbThisGene + " " + geneAttribute);
			String geneType = geneAttribute.getString("GeneClass");
			Class<? extends Gene> geneClass = (Class<? extends Gene>) Madkit.getClassLoader().loadClass(geneType);
			for (int i = 0; i < nbThisGene; i++) {
				Gene newGene = (Gene) (geneClass.newInstance());
				newGene.setAttrib(geneAttribute);
				newGene.init();
				add(0, newGene); // stack newGene in wrong way coz the
				// Iterator in createPool() give index in
				// wrong order.
			}
		} catch (ClassNotFoundException e) {
			System.err.println("Genome: gene class not found !\n" + e);
		} catch (IllegalAccessException e) {
			System.err.println("Genome: gene class access problem !\n" + e);
		} catch (InstantiationException e) {
			System.err.println("Genome: gene instantiation problem !\n" + e);
		}
	}

	/** add a gene in the genome */
	public boolean add(Gene newGene) {
		boolean result = super.add(newGene);
		refreshGui();
		return result;
	}
	
	/** initialisation. A waiting genome can be simulated in order to be selected */
	public void init() {
		state = WAITING;
	}

	/** reset the fitness value of the genome (i.e. after selection) */ 
	public void reset() {
		setFitness(0);
		state = WAITING;
		for (int i = 0; i < size(); i++) {
			if(!((Gene) get(i)).isFixed()) ((Gene) get(i)).init();
		}
		refreshGui();
	}

	/** clone method */
	public Object clone() {
		Genome newGenome = (Genome) super.clone();
		newGenome.clear();
		for (int i = 0; i < this.size(); i++) {
			newGenome.add(i, (Gene) get(i).clone());
		}
		if(fitnessLabel!= null) newGenome.setFitnessLabel(new JLabel(new Double(fitness).toString()));
		if(myGui != null) newGenome.setMyGui(new GenomeGUI(newGenome));
		return (Object) newGenome;
	}

	/** Given all the fitness values given by manager, this method evaluates the global fitness of the genome*/
	public void evaluateFitness() {
		fitness = 0;
		if (managerTable.size() > 0) {
			for (Double value : managerTable.values()) 
				fitness += value;				
			fitness /= managerTable.size();
		}
		refreshFitnessLabel();
	}

	/** toString method */
	public String toString() {
		String genome = ("Pool: " + poolName + "\n");
		genome += " fitness " + fitness + "\n";
		for (int i = 0; i < size(); i++) {
			genome += "Gene " + i + ": " + get(i);
		}
		// genome += "\n" + managerTable;
		// genome += "\n" + nbGenes + " " + fitness+ " " + poolName;
		return genome;
	}

	/** ***********Getters & * Setters***************** */
	public Gene get(String name) {
		for (int i = 0; i < size(); i++) {
			if (((Gene) get(i)).getName().equals(name))
				return (Gene) get(i);
		}
		return (Gene) get(0);
	}

	public ArrayList<? extends Gene> getAll(final String name) {
		ArrayList<Gene> genes = new ArrayList<Gene>();
		for (Gene gene : this) {
			if (name.equals(gene.getName())) {
				genes.add(gene);
			}
		}
		return genes;
	}

	public void setFitness(Manager myManager, double fitnessValue) {
		managerTable.put(myManager, new Double(fitnessValue));
		evaluateFitness();
	}

	public void setHadocFitness(double fitnessValue) {
		managerTable.put("user", new Double(fitnessValue));
		evaluateFitness();
	}

	/** Getter */
	public int getNbGenes() {
		return size();
	}

	public int getSize() {
		return size();
	}

	public double getFitness() {
		return fitness;
	}

	public void setFitness(double fit) {
		fitness = fit;
		refreshFitnessLabel();
	}

	public Gene getGene(int i) {
		return (Gene) get(i);
	}

	public String getPoolName() {
		return poolName;
	}

	public void setPoolName(String pool) {
		poolName = pool;
	}

	public String shortToString() {
		String output = "";
		for (int i = 0; i < size(); i++) {
			output += "[" + ((Gene) get(i)).getAttrib().getString("GeneClass")
					+ "]   ";
		}
		return output;
	}

	public int getState() {
		return state;
	}

	public void setState(int state) {
		this.state = state;
		refreshGui();
	}

	public void refreshGui() {
		if (myGui != null)
			myGui.refresh();
	}

	public void refreshFitnessLabel() {
		if (fitnessLabel != null)
			fitnessLabel.setText(new Double(fitness).toString());
	}

	public JLabel getFitnessLabel() {
		if (fitnessLabel == null) {
			fitnessLabel = new JLabel(new Double(fitness).toString());
		}
		return fitnessLabel;
	}

	public JPanel getPanel() {
		if (myGui == null)
			myGui = new GenomeGUI(this);
		return (JPanel) myGui;
	}

	public void setFitnessLabel(JLabel fitnessLabel) {
		this.fitnessLabel = fitnessLabel;
	}
	public void setMyGui(GenomeGUI g) {
		myGui = g;
	}
	
	public boolean isEqual(Genome genome){
		for (int i = 0; i < size(); i++) {
			if(!((Gene) get(i)).isEqual((Gene)genome.get(i))) return false;
		}
		return true;
	}
}
