/*
 * Turtle.java -TurtleKit - A 'star logo' in MadKit
 * Copyright (C) 2000-2007 Fabien Michel
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package turtlekit.kernel;

import java.awt.Color;
import java.lang.reflect.Method;
import java.util.ArrayList;

import madkit.kernel.AbstractAgent;
import madkit.kernel.ReferenceableAgent;


/** The Turtle class implements the Turtle commands which are used to move set heading... 

    @author Fabien MICHEL
    @version 1.4 19/03/2008 */

public class Turtle extends AbstractAgent implements ReferenceableAgent
{
	public static final String TURTLE_DEFAULT_ROLE = "turtle";
	/**
	 * 
	 */
	private static final long serialVersionUID = -560589112217896865L;
	/**default direction values: setHeading(East) ~ setHeading(0)*/
	static public final int East=0,NorthEast=45,North=90,NorthWest=135,West=180,SouthWest=225,South=270,SouthEast=315;
	private double angle,x,y,angleCos=1,angleSin=0;
	private int who;
	boolean hidden=false;
	private TurtleEnvironment world;
	private String initMethod;
	private ArrayList<String> rolePlayed;

	Method nextAction = null;
	Color color=Color.red;
	Patch position;

	/** the initMethod is the first action (after setup) that the turtle will do*/
	public Turtle()
	{
		initMethod="defaultAction";
		randomHeading();
	}


	//For the Python Mode
	public Turtle(final String initMethod)
	{
		this.initMethod=initMethod;
	}

	final void setNextAction(final Method nextMethod)
	{
		if (nextMethod == null)
			die();
		else
			nextAction= nextMethod;
	}

	final public String defaultAction(){return "defaultAction";}

	final void initialisation(final int a,final int b,final TurtleEnvironment w,final int t,final Patch pos)
	{
		world =w;
		try
		{
			final Method first = getClass().getMethod(initMethod);
			setNextAction(first);
		}
		catch (final Exception e) {System.err.println("Can't find method: "+initMethod+"\n");e.printStackTrace();}
		who = t;
		position = pos;
		x=a;
		y=b;
	}

	/**Madkit kernel usage : you must include super.activate() when overriding*/
	public void activate()
	{
		requestRole(Launcher.COMMUNITY,getSimulationGroup(),TURTLE_DEFAULT_ROLE,null);
		setup();
	}
	/**Madkit kernel usage : you should include super.end() when overriding*/
	public void end()
	{
		//leaveGroup(Launcher.COMMUNITY,getSimulationGroup());
	}

	public void setup(){};

	final public void die()	{
		if(position != null){
			world.removeTurtle(this); 
			position=null;
			rolePlayed = null;
		}
	}

//	final double normeX(final double a){
//	if (world.wrap)	
//	if (a>world.x)
//	return a%world.x;
//	else
//	if (a < 0) return a+world.x;
//	else return a;
//	else
//	if (a>=(world.x-0.5))
//	return world.x-1;
//	else
//	if (a<0) return 0;
//	else return a;
//	}

	final double normeValue(double a, final int worldThickness){
		if (world.wrap)	{
			a%=worldThickness;
			return a < 0 ? (a + worldThickness)%worldThickness : a;
		}
		else
			if (a >= worldThickness)
				return worldThickness-.01;
			else
				return a < 0 ? 0 : a;
	}

	final int normeValue(int a, final int worldThickness){
		if (world.wrap){	
			a %= worldThickness;
			return a < 0 ? a + worldThickness : a;
		}
		else
			if (a >= worldThickness)
				return worldThickness-1;
			else
				return a < 0 ? 0 : a;
	}

//	final int normeY(int a){
//	if (world.wrap)
//	{	
//	a %=world.y;
//	if (a < 0) return a+world.y;
//	else return a;
//	}
//	else
//	if (a>=world.y)
//	return world.y-1;
//	else
//	if (a<0) return 0;
//	else return a;
//	}


	public String toString(){return "turtle "+who+" at "+xcor()+" "+ycor()+" : heading="+angle+",color="+color;}

	///////////////////// the turtle command  /////////////////////////////////

	/**get the MadKit group of the simulation*/
	public String getSimulationGroup(){return world.simulationGroup;}

	/**one way to identify a kind of turtle: give them a Role in the simulation.*/
	public final void playRole(final String role)
	{
		if (rolePlayed == null) 
			rolePlayed = new ArrayList<String>();
		rolePlayed.add(role);
		requestRole(Launcher.COMMUNITY,getSimulationGroup(),role,null);
	}
	public final boolean isPlayingRole(final String role)
	{
		return (rolePlayed != null && rolePlayed.contains(role));
	}
	/**the turtle will no longer play the specified role*/
	public final void giveUpRole(final String role)
	{
		leaveRole(getSimulationGroup(),role);
		if (rolePlayed != null)
			rolePlayed.remove(role);
	}
	/**return the current heading of the turtle*/
	public final double getHeading(){return angle;}
	/**set the turtle heading to the value of direction*/
	public final void setHeading(final double direction)
	{
		angle = direction%360;
		if (angle < 0) angle+=360;
		angleSin=Math.sin( (Math.PI*angle)/180);
		angleCos=Math.cos( (Math.PI*angle)/180);
	}

	public final void setColor(final Color c){color=c;}
	public final Color getColor(){return color;}
	/**if true, the turtle hides itself (no draw)*/
	public final void setHidden(final boolean b){hidden = b;}
	public final boolean getHidden(){return hidden;}
	public final void setPatchColor(final Color c){ if (position != null) position.setColor(c);}
	public final Color getPatchColor(){return position != null ? position.color : null;}
	/**get the color of the patch situated at (a,b) units away*/
	public final Color getPatchColorAt(final int a,final int b){return world.getPatchColor(normeValue(a+xcor(),world.x),normeValue(b+ycor(),world.y));}
	/**set the color of the patch situated at (a,b) units away*/
	public final void setPatchColorAt(final Color c,final int a,final int b){if (position != null) world.setPatchColor(c,normeValue(a+xcor(),world.x),normeValue(b+ycor(),world.y));}

	/**turtle move forward*/
	public final void fd(final int nb)
	{
		moveTo(x+angleCos*nb,y+angleSin*nb);

		/*
	for(int i = 0;i < nb;i++)
	    {
		x = normeX(x+angleCos);
		y = normeY(y+angleSin);
		world.moveTurtle(x,y,this);
	    }*/
	}
	/** turtle move backward*/
	public final void bk(final int nb)
	{
		moveTo(x-angleCos*nb,y-angleSin*nb);
		/*		x = normeX(x-angleCos*nb);
		y = normeY(y-angleSin*nb);
		world.moveTurtle(x,y,this);
/*	for(int i = 0;i < nb;i++)
	    {
		x = normeX(x-angleCos);
		y = normeY(y-angleSin);
		world.moveTurtle(x,y,this);
	    }*/
	}
	/** transfers the turtle to patch (a,b).
	Can be used as a jump primitive: MoveTo(xcor()+10,ycor())*/
	/*public final void moveTo(int a,int b)
    {
	x = normeX(a);
	y = normeY(b);
	world.moveTurtle(x,y,this);
    }*/

	/** transfers the turtle to patch (a,b).
	Can be used as a jump primitive: MoveTo(xcor()+10,ycor())*/
	public final void moveTo(final double a,final double b)
	{
		if (position != null) {
			x = normeValue(a,world.x);
			y = normeValue(b,world.y);
			world.moveTurtle(xcor(),ycor(),this);
		}
	}

	public final void moveTo(int a, int b)
	{
		if (position != null) {
			a = normeValue(a,world.x);
			b = normeValue(b,world.y);
			x = a;
			y = b;
			world.moveTurtle(a, b, this);
		}
	}

	/**transfers the turtle to the center patch*/ 
	public final void home(){
		if (position != null) {
			x = world.x / 2;
			y = world.y / 2;
			world.moveTurtle(xcor(), ycor(), this);
		}
	}

	public final void setX(final double a)
	{
		if (position != null) {
			x = normeValue(a,world.x);
			world.moveTurtle(xcor(), ycor(), this);
		}
	}
	public final void setY(final double b)
	{
		if (position != null) {
			y = normeValue(b,world.y);
			world.moveTurtle(xcor(), ycor(), this);
		}
	}
	public final void setXY(final double a,final double b)
	{
		if (position != null) {
			x = normeValue(a,world.x);
			y = normeValue(b,world.y);
			world.moveTurtle(xcor(), ycor(), this);
		}
	}

	/**
	 * return the "on screen distance" between the turtle and the patch of absolute coordinates (a,b).
	 * 
	 * @param a the a
	 * @param b the b
	 * 
	 * @return a distance using double
	 */
	final public double distanceNowrap(double a,double b)
	{
		a = normeValue(a,world.x);
		b = normeValue(b,world.y);
		a-=x;
		b-=y;
		return Math.sqrt( a*a + b*b );
	}
	
	/**
	 * returns the distance from the patch (a,b).
	 * The "wrapped distance", when wrap mode is on, (around the edges of the screen)
	 * if that distance is shorter than the "onscreen distance."
	 * 
	 * @param a the a
	 * @param b the b
	 * 
	 * @return the distance using double
	 */
	public final double distance(double a,double b)
	{
		if (! world.wrap) return distanceNowrap(a,b);
		a = normeValue(a,world.x);
		b = normeValue(b,world.y);
		if (Math.abs(a-x) > world.x/2)
			if (a < x) a+=world.x;
			else a-=world.x;
		if (Math.abs(b-y) > world.y/2)
			if (b < y) b+=world.y;
			else b=b-world.y;
		a-=x;
		b-=y;
		return Math.sqrt( a*a + b*b );
	}
	public final double towardsNowrap(double a,double b)
	{
		a = normeValue(a,world.x);
		b = normeValue(b,world.y);
		a-=x;
		b-=y;
		if (a == 0 && b == 0) return 0.0;
		if (b < 0)
			return 180*Math.asin(a / Math.sqrt(a*a+b*b))/Math.PI+270;
		else
			return 180*Math.acos(a / Math.sqrt(a*a+b*b))/Math.PI;
	}
	/**returns direction to the patch (a,b).
       If the "wrapped distance", when wrap mode is on, (around the edges of the screen)
       is shorter than the "onscreen distance," towards will report
       the direction of the wrapped path,
       otherwise it while will report the direction of the onscreen path*/
	public final double towards(double a,double b)
	{
		if (! world.wrap) return towardsNowrap(a,b);
		if (distance(a,b) > distanceNowrap(a,b))
			return towardsNowrap(a,b);
		else 
		{
			a = normeValue(a,world.x);
			b = normeValue(b,world.y);
			if (Math.abs(a-x) > world.x/2)
				if (a < x) a=a+world.x;
				else a-=world.x;
			if (Math.abs(b-y) > world.y/2)
				if (b < y) b=b+world.y;
				else b=b-world.y;
			a-=x;
			b-=y;
			if (a == 0 && b == 0) return 0.0;
			if (b < 0)
				return 180*Math.asin(a / Math.sqrt(a*a+b*b))/Math.PI+270;
			else
				return 180*Math.acos(a / Math.sqrt(a*a+b*b))/Math.PI;
		}
	}

	public final void randomHeading() { setHeading(Math.random()*360); }

	/**create a turtle at the creator position (xcor,ycor)
       returns the ID of the new turtle*/
	public final int createTurtle(final Turtle t){ return position != null ? world.addAgent(t,xcor(),ycor()) : -1;}

	public final int xcor(){ return (int) x; }
	public final int ycor(){ return (int) y; }

	public final double realX(){ return x; }
	public final double  realY(){ return y; }

	/**return the Turtle with the specified ID, null if not alive*/
	public final Turtle getTurtleWithID(final int id){
		return world.getTurtleWithID(id);
	}

	/**return the x-increment if the turtle were to take one
       step forward in its current heading.*/
	public final int dx()
	{
		return ((int) (x+angleCos)) - ((int) x);
	}
	/**return the y-increment if the turtle were to take one
       step forward in its current heading.*/
	public final int dy()
	{
		return ((int) (y+angleSin)) - ((int) y);
	}

	public final void turnRight(final double a){angle-=a;	setHeading(angle);}
	public final void turnLeft(final double a){angle+=a;	setHeading(angle);}

	/**@return the other turtles on the current patch*/
	public final Turtle[] turtlesHere() {return position != null ? position.getOtherTurtles(this) : null;}
	/**return turtles who are on the patch situated at (a,b) units away*/
	public final Turtle[] turtlesAt(final int a,final int b){return position != null ? world.turtlesAt(normeValue(a+xcor(),world.x),normeValue(b+ycor(),world.y)) : new Turtle[0];}

	public final int countTurtlesHere(){return position != null ? position.turtlesHere.size() : null;}
	/**return the number of turtles in the patch situated at (a,b) units away*/
	public final int countTurtlesAt(final int a,final int b){return world.turtlesCountAt(normeValue(a+xcor(),world.x),normeValue(b+ycor(),world.y));}

	/**return the turtle ID*/
	public final int mySelf(){return who;}

	public final int getWorldWidth() {return world.x;}
	public final int getWorldHeight(){return world.y;}

	/**return the value of the corresponding patch variable*/
	public final double getPatchVariable(final String variableName){return position != null ? position.getVariableValue(variableName) : 0;}
	/**return the value of the patch situated at (a,b) units away*/
	public final double getPatchVariableAt(final String variableName,final int a,final int b){return world.grid[normeValue(a+xcor(),world.x)][normeValue(b+ycor(),world.y)].getVariableValue(variableName);}
	/**set the value of the corresponding patch variable*/
	public final void incrementPatchVariable(final String variableName,final double value){if(position!=null)position.incrementPatchVariable(variableName,value);}
	public final void incrementPatchVariableAt(final String variableName,final double value,final int a,final int b){if(position!=null) world.grid[normeValue(a+xcor(),world.x)][normeValue(b+ycor(),world.y)].incrementPatchVariable(variableName,value);}
	/** get a mark deposed on the patch
	@return the corresponding java object, null if not present*/
	public final Object getMark(final String variableName){return position != null ? position.getMark(variableName) : null;}
	public final Object getMarkAt(final String variableName,final int a,final int b){return position != null ? world.grid[normeValue(a+xcor(),world.x)][normeValue(b+ycor(),world.y)].getMark(variableName) : null;}
	/** Drop a mark on the patch
	@param markName: mark name
	@param theMark: mark itself, can be any java object*/
	public final void dropMark(final String markName,final Object theMark){if(position!=null)position.dropMark(markName,theMark);}
	final public void dropMarkAt(final String markName,final Object theMark,final int a,final int b){if(position!=null) world.grid[normeValue(a+xcor(),world.x)][normeValue(b+ycor(),world.y)].dropMark(markName,theMark);}
	/** test if the corresponding mark is present on the patch (true or false)*/
	public final boolean isMarkPresent(final String markName){return position != null ? position.isMarkPresent(markName) : null;}  
	/** test if the corresponding mark is present on the patch situated at (a,b) units away*/
	public final boolean isMarkPresentAt(final String markName,final int a,final int b){return world.grid[normeValue(a+xcor(),world.x)][normeValue(b+ycor(),world.y)].isMarkPresent(markName);}  
	//////////////////////////////////////////////
}
