
package modulecoFramework.simulation;

import modulecoFramework.modeleco.EWorld;
import modulecoFramework.modeleco.WorldListener;
import modulecoFramework.modeleco.ZoneSelector;
import modulecoFramework.simulation.scheduling.ModulecoScheduler;
import madkit.kernel.*;
import modulecoFramework.Moduleco;

/**
 * Copyright (c)enst-bretagne & Lepii
 * @author denis.phan@enst_bretagne.fr - Modify by Thibaud Roussillat
 * @version 1.5.1a - Last modify on 20 September 2005
 * <br>
 * Run the simulation by managing the time scheduler.
 */

public class SimulationControl extends Agent {
    
	
	/** serialVersionUID **/
	private static final long serialVersionUID = -512994355855853403L;

    transient protected boolean runnable = false;
    
    /** The current world. **/
    protected EWorld eWorld;
   
    /**
     * The TimeScheduler ts is the evolution strategy adopted by agents. <br>
     * e.g. Early or Late Commit or Event Scheduling.
     */
    protected ModulecoScheduler ts;
    
    /**
     * The ZoneSelector zs describes the evolving zone of agents. <br>
     * e.g. Moore
     */
    protected ZoneSelector zs;
    
    /**
     * The time step.
     */
    protected int iter = 0;
    
    /**
     * The number of steps during the simulation. Can be modified by ModulecoAgent if 
     * this number is specified for batches
     */
    private int simulationDuration=Integer.MAX_VALUE;
    
    /**
     * The name of the classes chosen as TimeScheduler and ZoneSelector. Used to
     * create a newInstance()
     */
    public String tsClass, zsClass;
    
    /**
     * A WorldListener to interact between the World and the CentralControl,
     * i.e. between the Framework and the GUI.
     */
    public WorldListener worldListener;
    
    
    /** 
     * Constructeur du SimulationControl :
     * @param eWorld : the world
     * @param worldListener : the worldListener
     * @param zsClass : the class name of the zone selector
     */
    public SimulationControl(EWorld eWorld, WorldListener worldListener, String zsClass) {
        /* Affectation des paramtres */
    	this.eWorld = eWorld;
        this.worldListener = worldListener;
        this.zsClass = zsClass;
    }
    
    /**
     * Build the TimeScheduler and ZoneSelector.
     */
    public void buildScheduler() {
    	
    	/* Affectation du simulationControl au eWorld car celui-ci peut dans certain cas (par exemple
    	 * dans le modle smallWorld) arret de lui-mme la simulation */
        eWorld.setSimulationControl(this);
        
        try {
            /* Define selector and scheduler after populateAll(nsClass) since
             * some lists of agents need the population to be created. */
            zs = (ZoneSelector) Moduleco.getClass(zsClass).newInstance();
            zs.setWorld(eWorld);
            
            /* Cration du Scheduler et lancement en tant qu'agent Madkit */
            ts = new ModulecoScheduler();
            launchAgent(ts,"ModulecoScheduler",false);
            /* Configuration du zoneSelector du scheduler */
            ts.setZoneSelector(zs);
            
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
    
    /**
     * Initialise the simulation. Call : 
     * <ul>
     * <li>call eWorld.connectAll()</li>
     * <li>call eWorld.initAll()</li>
     * <li>call eWorld.compute()</li>
     * </ul>
     */
    public void initSimulation() {

        /* Connect Agents to mediums. */
        eWorld.connectAll();
        
        /* Initialize the agents, the mediums and finally the world. */
        eWorld.initAll();
        
        /* Compute the initial global state. */
        eWorld.compute();
        
    }
    
    /**
     * Make the world evolve one step. <br>
     * Called by the method this.live() or by centralControl if the user use the step by step function
     */
    public void progress() {
        /** Makes the simulation evolve one event.*/
        ts.step();
    }
    
    /**
     * Manage the simulation thread.
     */
    public void live() {
        while(true){
            exitImmediatlyOnKill();
            while (!runnable) {
                exitImmediatlyOnKill();
                pause(10);
            }
            /**
             * Make the world progress one step.
             */
            Thread.yield();
            iter=(int)ModulecoScheduler.getGVT();
            if(getSimulationDuration()>iter){
            	pause(1);
            	progress();
            }
            else{
                sendMessage(Moduleco.COMMUNITY,Moduleco.ENGINE_GROUP,"modulecoAgent", new Message());
                runnable=false;
            }
        }
    }
    
    /**
     * Start the simulation. Invoked by ModulecoAgent.live() for batch 
     * or by JToolBarCommand.actionPerformed() through CentralControl.simulationStart() if Gui is used:
     */
    public void start() {
    	/* Debloque la boucle situ dans la mthode live du thread */
    	runnable = true;
    }
    
    /**
     * Interrupt the Thread but the simulation is not closed(). <br>
     * Invoked by JToolBarCommand.actionPerformed()
     * through CentralControl.simulationStop()
     *
     * @see modulecoGUI.CentralControl.simulationStop()
     */
    public void stop() {
    	/* Bloque la boucle situ dans la mthode live du thread */
        runnable = false;
    }
    
    /**
     * Close the current simulation. <br>
     * Invoked by modulecoGUI.CentralControl.simulationTerminate() by
     * encapsulation through CentralControl.simulationTerminate()
     * or by ModulecoLauncher.reload()
     */
    public void terminate() {
    	killAgent(ts);
        killAgent(this);
        eWorld.terminate();
    }
    
    /**
     * Get the time step.
     * @return iter
     */
    public int getIter() {
        return iter;
    }
    
    /** 
     * Appel au lancement de l'agent et demande le rle "simulationControl" au sein 
     * du groupe Moduleco.ENGINE_GROUP
     */
    public void activate(){
        requestRole(Moduleco.COMMUNITY, Moduleco.ENGINE_GROUP,"simulationControl", null);
    }
    
    /**
     * Termine le thread et l'agent
     */
    public void end(){
        System.err.println(""+this.toString()+" ending !!");
    }

    /** 
     * 
     * @return the number of maximum step of the simulation 
     */
    public int getSimulationDuration() {
        return simulationDuration;
    }

    /**
     * Used in batch mode for the moment
     * @param simulationDuration : the number of maximum step of the simulation
     */
    public void setSimulationDuration(int simulationDuration) {
        this.simulationDuration = simulationDuration;
    }
}