
package modulecoFramework.simulation;

import java.io.File;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;

import javax.swing.JPanel;

import modulecoGUI.CentralControl;

import madkit.kernel.*;
import madkit.system.MadkitOutput;

import modulecoFramework.Moduleco;
import modulecoFramework.modeleco.EWorld;
import modulecoFramework.modeleco.ModelParameters;
import modulecoFramework.modeleco.WorldListener;
import modulecoFramework.utils.ModelReader;
import modulecoFramework.utils.ModelWriter;
import modulecoFramework.utils.dataRecorder.Recorder;
import modulecoFramework.utils.dataRecorder.StatRecorder;
import modulecoFramework.utils.graphicalWatcher.GraphicalWatcher;

/**
 * Copyright (c)enst-bretagne & Lepii
 * @author denis.phan@enst_bretagne.fr modify by Thibaud Roussillat
 * @version 1.5.1a - Last modify on 20 September 2005
 * 
 * Point d'entre du noyau de simulation par rapport  l'interface graphique.
 * Les ordres provenant du centralControl sont  destination de cette classe.<br>
 * Launch the world. <br>
 * The option with no GUI is usefull in case we want to use the command-line.
 * 
 */

public class ModulecoLauncher extends AbstractAgent{
    
	/** serialVersionUID **/
	private static final long serialVersionUID = 1L;
	
	/** The CentralControl is used to managed the GUI **/
    protected CentralControl centralControl;
    
    /** The SimulationControl runs the simulation by managing the time scheduler. **/
    protected SimulationControl simulationControl;
    
    /** The World. **/
    protected EWorld eWorld;
   
    /** The model's name **/
    protected String model;
    
    /** Parameters of the model **/
    protected ModelParameters modelParameters;
    
    /** Listen to the world during a simultion. **/
    protected WorldListener worldListener;
    
    /** Permet d'observer dynamiquement les agents... **/
    GraphicalWatcher graphicalWatcher;
    
    /** Permet d'enregistrer les donnes relatives aux agents (donnes choisi par l'utilisateur)**/
    Recorder recorder;
    
    /** Permet d'enregistrer les donnes du monde pour les rendres accessible aux graphiques **/
    StatRecorder statRecorder;
    
    /** Console... **/
    MadkitOutput madkitOutput;
    
    /** Constructor called from ModulecoAgent **/ 
    public ModulecoLauncher() {
       
    }
    
    /**
     * Appel par ModulecoAgent qui a construit le CentralControl si l'on est en mode graphique
     * <br>
     * We now have a reference to the CentralControl.
     * @param : the CentralControl built by ModulecoAgent
     */
    public void setCentralControl(CentralControl centralControl) {
        this.centralControl = centralControl;
    }
    
    /**
     * Create a world from the model name and eventually the parameters provided
     * (usually by the GUI). <br>
     * We provide parameters only when we re-create the same world. If we don't,
     * parameters will be read in the World class by using reflection.
     *
     * @param modelName : name of the model provided by the GUI or by the config file if we use batch
     */
    public void create(String modelName) {
         
    	/* Rcupration du nom du modle prcdent */
    	String oldName = Moduleco.getCurrentModelName();
    	
    	/* Mise  jour du nom du modle actuel */
        Moduleco.setCurrentModelName(modelName);
        
        /* Reset le ModulecoLaucher en tuant les agents de la simulation
         * prcdent si elle existait */
        this.reset();
        
        /* Constructeur du monde */
        Constructor worldConstructor;
        
        try {
           
        	/* Rcupration de la classe du monde situ dans le jar du modle et
        	 * via le class Loader de MadKit */
            Class worldClass = Moduleco.getClass("models." + modelName + ".World");
            
            /* Rcupration de l'objet constructor (qui en paramtre 0 arguments) */
            worldConstructor = worldClass.getConstructor(new Class[] {});  
            
            try {
            	/* Construit un nouveau monde en crant une instance de la classe World */
            	eWorld = (EWorld) worldConstructor.newInstance(new Object[]{});
            } catch (InvocationTargetException e) {
            	System.out.println(e.getTargetException().toString());
            }
            
            try {
            	/** TODO : BE CARREFUL !!! Le fichier temporaire devra tre charg par la suite
            	 * depuis ./temp/nomModel.parameters et le fichier de paramtres par dfaut depuis 
            	 * le jar du modle. Ces modifications obligerons par contre le concepteur  recrer
            	 * le jar mme s'il change juste le fichier de paramtre par dfaut. 
            	 */
            	
            	/* Rcupre le chemin ou est situ le modle (et non son jar) */
	            String path = eWorld.getClass().getPackage().getName().replace(".",File.separator);
	            
	            /* S'il on recharge le mme modle */
	            if (modelName.equals(oldName)) {
	            	/* Charge le fichier de paramtres temporaires correspondant  la simulation prcdente */
	            	this.readModelParameters(new File(path+File.separator+"temp.parameters"));
	            }
	            else {
	            	/* Sinon charge le fichier de paramtre par dfaut situ  l'emplacement indiqu */
	            	this.readModelParameters(new File(path+File.separator+modelName+".parameters"));
	            }
            }
	        catch (Exception e) {
	        	System.out.println("BE CARREFUL : Moduleco can't find parameters' file for model "+Moduleco.getCurrentModelName());
	        }
            
	        /* rcupre les paramtres du modle dans le cas o ils seraient null */
            modelParameters = eWorld.getModelParameters();
            if (modelParameters == null) {
            	/* Construction et affectation de paramtres par dfaut... */
            	modelParameters = new ModelParameters(modelName);
            	eWorld.setModelParameters(modelParameters);
            }
                 
            /* Launch the world as a Madkit Agent */
            launchAgent(eWorld,"world",false);
            
            /* Appel la mthode worldListener */
            this.createWorldListener();
            
        } 
        catch (Exception e) {
            System.out.println("[ModulecoLauncher.create()] Exception: " + e.toString());
            e.printStackTrace();
        }
    }
    
    /**
     * Create the WorldListener and eventually link it to the CentralControl, if
     * there is a GUI.
     */
    protected void createWorldListener() {
        
    	/* Cr un nouveau WorldListener auquel on passe en paramtre le monde, le modelParameters et le
    	 * centralControl */
        worldListener = new WorldListener(eWorld, modelParameters, centralControl);
        /* Launch the WorldListener as a MadKit Agent */
        launchAgent(worldListener,"world listener",false);
        
    }
    
    /**
     * Prepare le dpart d'une nouvelle simulation
     * Mthode notamment appel par ModulecoAgent dans le cas d'une simulation par batch 
     * ou par le centralControl lors du chargement d'un autre modle
     *
     */
    public void reload() {
    	simulationControl.terminate();
    	this.reset();
    }
    
    /**
     * Lance le monde quand les paramtres ont t spcifi via l'interface graphique 
     * (JPanelParameters > CentralControl > ModulecoLauncher ) 
     * s'il existe ou sinon via la mthode live() de ModulecoAgent
     */
    public void launch() {
    	
    	/* Affectation des paramtres au worldListener au cas o ils aient t modifi */
    	worldListener.setModelParameters(modelParameters);
    	
    	/* Rcupration du nom de classe complet du voisinage choisi */
    	String nsClass = "modulecoFramework.modeleco.zone."+modelParameters.getNeighbourhood();
    	
    	/* Rcupration du nom de classe complet de la zone de scheduling choisi */
    	String zsClass = "modulecoFramework.modeleco.zone."+modelParameters.getZone();
    	
    	/* Prparation du monde  son peuplement */
    	eWorld.build();
    	/* Prparation du monde  d'eventuelles connections */
    	eWorld.initializeMedium();
    	
    	/* Peuplement du monde, le nom de classe dfinissant le voisinage est pass en paramtre */
    	eWorld.populateAll(nsClass);
        
    	/* Construction d'un nouveau SimulationControl et lancement en tant qu'agent madkit */
    	simulationControl = new SimulationControl(eWorld, worldListener, zsClass);
        launchAgent(simulationControl,"controler",false);
        
        /* Construction du Scheduler et initialisation de la simulation  partir du simulationControl */
        simulationControl.buildScheduler();
        simulationControl.initSimulation();
        
        /** TODO : BE CARREFUL !!! Le chemin spcifi ici devra tre remplac par ./temp/nomModel.parameters **/
        /* Rcupration du chemin o est stock le modle et criture du fichier de paramtres temporaire */
        String path = eWorld.getClass().getPackage().getName().replace(".",File.separator);
        this.writeModelParameters(new File(path+File.separator+"temp.parameters"));
        
        /* Lance les modules d'enregistrement des donnes de simulation */
        this.runNewGraphicalWatcher();
        this.record();
        
    }
    
    /**
     * @return Returns the current eWorld.
     */
    public EWorld getEWorld() {
        return eWorld;
    }
    /**
     * @return Returns the simulationControl.
     */
    public SimulationControl getSimulationControl() {
        return simulationControl;
    }
    /**
     * @return Returns the modelParameters.
     */
    public ModelParameters getModelParameters() {
        return modelParameters;
    }
    
    /** 
     * Mthode appele lors du lancement de l'agent ModulecoLauncher  partir 
     * de la mthode activate() de ModulecoAgent.
     */
    public void activate(){
    	
    	/* trace */
        println("activated");
        
        /* Demande d'appartenance  un groupe et un rle */
        this.requestRole(Moduleco.COMMUNITY, Moduleco.ENGINE_GROUP,"modulecoLauncher", null);
        
        /** 
         * TODO :  grer par la suite pour charger la console Madkit (interception des sorties java) 
         * dans l'interface graphique 
         */
        /*
        madkitOutput = new MadkitOutput();
        this.launchAgent(madkitOutput,"Madkit Output",false);
        
        madkitOutput.initGUI();
        System.out.println("Output gui : " + madkitOutput.getGUIObject().getClass().getSuperclass().getName());
        */
    }
    
    /**
     * 
     * @return the GUI Object of the console
     */
    public JPanel getOutputGUI() {
    	// Si la console n'est pas lanc en tant qu'agent on renvoie null
    	if (madkitOutput != null) {
    		/* Recupre l'objet graphique de la console et le renvoie */
    		return (JPanel) madkitOutput.getGUIObject();
    	}
    	else {
    		return null;
    	}
    }
    
    /**
     * Mthode appel  la fin d'une simulation par le centralControl dans la mthode
     * reload()
     */
    public void end(){
        this.reset();
        println("ended");
    }
    
    /**
     * Tue tous les agents relatifs  la simulation courante : eWorld, simulationControl, 
     * graphicalWatcher, recorder & worldListener
     */
    public void reset(){
        killAgent(eWorld);
        killAgent(simulationControl);
        killAgent(graphicalWatcher);
        killAgent(recorder);
        killAgent(worldListener);
    }
    
    /**
     * Ecrit les paramtres du Modle courant dans un fichier xml dfinit par le paramtre file
     * @param file : le fichier xml o les paramtres doivent tre crit
     */
    public void writeModelParameters(File file) {
    	ModelWriter modelRecorder = new ModelWriter(eWorld);
    	modelRecorder.record(file);
    }
	
	/**
	 * Lit les paramtres situ dans le fichier xml file et les affecte au monde courant
	 * @param file : le fichier xml de paramtres
	 */
	public void readModelParameters(File file) {
		ModelReader modelReader = new ModelReader(eWorld);
    	modelReader.read(file);
	}
	
	/**
	 * Run a new GraphicalWatcher. Lanc au dbut de chaque nouvelle simulation et permet
	 * notamment d'enregistrer dynamiquement les donnes ncessaires aux graphiques
	 * crs par l'utilsiateur.
	 */
	public void runNewGraphicalWatcher() {
		
		/* Construction de l'objet */
		graphicalWatcher = new GraphicalWatcher();
		/* Lancement en tant qu'agent Madkit */
        launchAgent(graphicalWatcher,"Watcher",false);
		
	}
	
	/**
	 * Retourne le GraphicalWatcher courant. Peut tre ncessaire lors de la cration de graphiques dynamiques
	 * @return le GraphicalWatcher courant
	 */
	public GraphicalWatcher getGraphicalWatcher() {
		return this.graphicalWatcher;
	}
	
	/**
	 * Pour conomiser des cliques pr les tests...
	 * @param string : le nom du modle
	 */
	public void loadDefaultModel(String string) {
		centralControl.loadDefaultModel("seminaire");
	}
	
	/** 
	 * Lance les procdures d'enregistrement des donnes...<br>
	 * <ul>
	 * <li>Pour les enregistrer dans le fichier csv aprs choix de l'utilisateur dans l'interface</li>
	 * <li>Pour les mettres  disposition des graphiques statiques</li>
	 * </ul>
	 */
	protected void record() {
		
		/* Cr et lance un nouveau Recorder servant  enregistrer les donnes choisi par l'utilisateur
		 * dans le fichier csv */
		recorder = new Recorder();
		this.launchAgent(recorder,"Recorder",false);
		/* Demande d'enregistrer toutes les propriets du monde et des agents par dfaut */
		recorder.recordAllAgentProperty();
		recorder.recordAllWorldProperty();
		/* Lance le premier pas de l'observation pour avoir les donnes au pas de temps 0 */
		recorder.observeAgents();
		
		/* Cr et Lance un nouveau StatRecorder de l'enregistrement des variables du monde
		 * pour les mettre  disposition des graphiques statiques */
		statRecorder = new StatRecorder();
		this.launchAgent(statRecorder,"StatRecorder",false);
		/* Lance le premier pas de l'observation pour avoir les donnes au pas de temps 0 */
		statRecorder.observeAgents();
		
	}
	
	/**
	 * Retourne le Recorder courant qui permet d'enregistrer les donnes choisi par l'utilisateur
	 * @return the current Recorder
	 */
	public Recorder getRecorder() {
		return this.recorder;
	}

	/**
	 * Retourne le StatRecorder courant qui permet d'enregistrer les variables du monde
	 * pour les mettre  disposition des graphiques statiques
	 * @return the current StatRecorder
	 */
	public StatRecorder getStatRecorder() {
		return this.statRecorder;
	}
    
}