/*
 * SynchroScheduler.java - Kernel: the kernel of MadKit
 * Copyright (C) 1998-2008 Olivier Gutknecht, Fabien Michel, Jacques Ferber
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package madkit.kernel;



import java.util.ArrayList;
import java.util.Iterator;



/** This class is the non threaded version of a Scheduler.
 *  Don't forget to make it implementing the ReferenceableAgent interface if you plan to schedule it with activators.
 *
 *    @author Fabien MICHEL
 *    @version 1.0 05/09/01
 *    @since MadKit 3.1
 */



public abstract class SynchroScheduler extends AbstractAgent
{
	private ArrayList<Activator<? extends AbstractAgent>> activators = new ArrayList<Activator<? extends AbstractAgent>> (7);

	synchronized public void addActivator(Activator<? extends AbstractAgent> a)
	{
		if(currentKernel.addOverlooker(getAddress(), a, null))
			activators.add(a);
	}

	synchronized public void addActivator(Activator<? extends AbstractAgent> a, Object accessCard)
	{
		if(currentKernel.addOverlooker(getAddress(), a, accessCard))
			activators.add(a);
	}

	synchronized public void removeActivator(Activator<? extends AbstractAgent> a)
	{
		if(activators.remove(a))
			currentKernel.removeOverlooker(a);
	}

	@SuppressWarnings("unchecked")
	final public Activator<? extends AbstractAgent>[] getActivators()
	{
		return(Activator<? extends AbstractAgent>[]) activators.toArray();
	}

	/**
	 * Removes all the activators.
	 */
	synchronized public void removeAllActivators()
	{
		for (Iterator<Activator<? extends AbstractAgent>> i = activators.iterator();i.hasNext();)
		{
			currentKernel.removeOverlooker(i.next());
			i.remove();
		}
	}

	/** the end method will remove all the activators which have been used by the scheduler.
	 * 
	 *   Overriding this method, one should use <code>super.end()</code> first in order to ease the removal of simulated agents which will be killed. 
	 *   Doing so can greatly improved the "killing time" of simulation engines
	 * @see madkit.kernel.AbstractAgent#end()
	 */
	@Override
	public void end()
	{
		removeAllActivators();
	}

}

