/*
* Scheduler.java - Kernel: the kernel of MadKit
* Copyright (C) 1998-2008 Olivier Gutknecht, Fabien Michel, Jacques Ferber
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 2.1 of the License, or (at your option) any later version.

* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.

* You should have received a copy of the GNU Lesser General Public
* License along with this library; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
package madkit.kernel;

import java.util.ArrayList;
import java.util.Iterator;

/** This class defines a generic threaded scheduler agent. It holds a collection of activators.
    @author Fabien Michel (MadKit 3.0 05/09/01) and 2.0 (Overlooker).
    @author Olivier Gutknecht version 1.0
    @since MadKit 2.0
    @version 3.0
*/

public abstract class Scheduler extends Agent
{
	private ArrayList<Activator<? extends AbstractAgent>> activators = new ArrayList<Activator<? extends AbstractAgent>> (7);
	
public void addActivator(Activator<? extends AbstractAgent> activator)
{
	if(currentKernel.addOverlooker(getAddress(), activator, null))
		activators.add(activator);
}

public void addActivator(Activator<? extends AbstractAgent> activator, Object accessCard)
{
	if(currentKernel.addOverlooker(getAddress(), activator, accessCard))
		activators.add(activator);
}
 
public void removeActivator(Activator<? extends AbstractAgent> activator)
{
	if(activators.remove(activator))
		currentKernel.removeOverlooker(activator);
}

/**
 * Gets the activators.
 * 
 * @return the activators
 */
@SuppressWarnings("unchecked")
final public Activator<? extends AbstractAgent>[] getActivators()
{
	return (Activator<? extends AbstractAgent>[]) activators.toArray();
}

public void removeAllActivators()
{
	for( Activator<? extends AbstractAgent> i : activators ){
		currentKernel.removeOverlooker(i);
		if(getDebug())
			println("removing activator "+i);
	}
	activators.clear();
}

/** the end method will remove all the activators which have been used by the scheduler.
 * 
 *   Overriding this method, one should use <code>super.end()</code> first in order to ease the removal of simulated agents which will be killed. 
 *   Doing so can greatly improved the "killing time" of simulation engines
 * @see madkit.kernel.AbstractAgent#end()
 */
@Override
public void end()
{
	removeAllActivators();
	activators = null;
}

}
