/*
 * Message.java - Kernel: the kernel of MadKit
 * Copyright (C) 1998-2008 Olivier Gutknecht, Fabien Michel, Jacques Ferber
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package madkit.kernel;

import java.util.Date;

/** The generic MadKit message class. Use a subclass to adapt it to
  * your needs. This class is quite lightweight, it just defines sender
  * and receivers (expressed with AgentAddress class) and automatically
  * stamp the date of send on the message. 
    
  * If security is an issue for the application, implementing the <code>SecuredMessage</code> interface should be considered as,
  * although the default fields of a message could not be altered by receivers (read only or copy),
  * other messages (subclasses of Message) and their content could be intercepted via the hooking mechanism of the kernel.
  * Thus, additional content could be altered by receivers if not read only.
  * Implementing the <code>SecuredMessage</code> interface 
  * (and Overriding the default <code>clone</code> method of <code>Message</code>; Do not forget to call super.clone())
  * will tell the kernel
  * to do a copy of the message for each receiver, ensuring the data security of the original object/content used to build a message.
  * @see SecuredMessage
  * @version 3.1
  * @author Olivier Gutknecht
  * @author Fabien Michel 
 */

@SuppressWarnings("serial")
public class Message implements java.io.Serializable, Cloneable
{
	final private long creationDate;
	private AgentAddress sender=null;
	private AgentAddress receiver=null;

	/**
	 * Instantiates a new empty message.
	 */
	public Message() {
		creationDate=System.currentTimeMillis();
	}
	
	final void setSender(AgentAddress s) {sender=s;}

	final void setReceiver(AgentAddress s) {receiver=s;}

	/**
	 * @return the original sender of this message.
	 */
	final public AgentAddress getSender() {return sender;}

	/**
	 * @return the intended receiver.
	 */
	final public AgentAddress getReceiver() {return receiver;}

	/**
	 * Gets the creation date.
	 * 
	 * @return the creation date for the message (instantiation time)
	 */
	final public java.util.Date getCreationDate() {return new Date(creationDate);}

	/**
	 * @return a shallow copy of the message.
	 */
	@Override
	public Message clone() {
		try	{
			return (Message) super.clone();
		}
		catch (CloneNotSupportedException e) { 
			e.printStackTrace();
			throw new InternalError(); 
		}
	}

	/** Returns a debug string with envelope and content for the message */
	@Override
	public String toString() {
		return "\nSender:"+sender+"\n"+" Receiver:"+receiver+"\n"+" Creation Date:"+new Date(creationDate)+"\n"+" Class:"+getClass().getName();
	}		

}