/*
 * Group.java - Kernel: the kernel of MadKit
 * Copyright (C) 1998-2008 Olivier Gutknecht, Fabien Michel, Jacques Ferber
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package madkit.kernel;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.Vector;

/**
 * A MadKit group definition. A group in Aalaadin/MadKit is defined as a set
 * of Role objects. Any member can hold many roles within the group.
 * This class is an internal MadKit structure.
 * 
 * new Group class for MadKit 3.0
 * 
 * @author Fabien Michel
 * @version 3.2 01/10/07
 */
@SuppressWarnings("serial")
final class Group extends HashMap<String,Role>
{
	private static final String GROUP_MANAGER = "group manager".intern();
	static final String MEMBER_DEFAULT_ROLE = "member".intern();
	private boolean distributed=false;
	private String description=null;
	private GroupIdentifier groupGate=null;
	//private Map<AgentAddress,Set<Role>> agentsToRole;

	/**this constructor assumes the creation of an empty group which is overlooked 
 We can say a virtual group, i.e. with no member.
 (by default, in the system the resulting group will not be distributed and description is null.
 but this will be changed during the real creation)*/
	Group(){
	}

	/**this constructor is invoked when one agent success in creating a group*/
	Group(final AgentAddress creator, final boolean distributed,final String description, final GroupIdentifier theIdentifier)
	{
		realCreation(creator,  distributed, description,  theIdentifier);
	}

	void realCreation(final AgentAddress creator, final boolean distributed, final String description, final GroupIdentifier theIdentifier)
	{
		if(containsKey(GROUP_MANAGER)) //in case this role is overlooked
		{
			AgentAddress ancienCreator = null;
			try {
				ancienCreator = getRolePlayer(GROUP_MANAGER);
			} catch (CGRException e) {
			}
			if(ancienCreator != null)
				get(GROUP_MANAGER).removeMember(ancienCreator);
			get(GROUP_MANAGER).addMember(creator);
		}
		else
		{
			final Role groupManager = new Role(1);	// à optimiser
			groupManager.addMember(creator);
			put(GROUP_MANAGER,groupManager);
		}
		if(containsKey(MEMBER_DEFAULT_ROLE)) //in case this role is overlooked
			get(MEMBER_DEFAULT_ROLE).addMember(creator);
		else
		{
			Role members;
//			if(Kernel.agressiveHeapMode)
//				members = new Role(Kernel.defaultAgentsAllocation);
//			else
				members = new Role();
			members.addMember(creator);
			put(MEMBER_DEFAULT_ROLE,members);
		}
		this.distributed = distributed;
		this.description = description;
		groupGate = theIdentifier;
	}


///////////////////////////////////// 	FUNDAMENTAL OPERATIONS 	ON GROUP : REQUESTRole, LEAVEROLE & LEAVEGROUP
	private int roleAssignment(final AgentAddress requester,final String roleName) throws CGRException
	{
		Role theRole = get(roleName);
		if(theRole != null)
		{
			if( theRole.addMember(requester) )
			{
				//get(MEMBER_DEFAULT_ROLE).addMember(requester);
				return 1;	//operation success
			}
			else
				throw new CGRException(-4,null,null,roleName,new RoleAlreadyHandled());	//role already handled
		}
		else
		{
			roleName.intern();
//			if(Kernel.agressiveHeapMode)
//				theRole = new Role(Kernel.defaultAgentsAllocation);
//			else
//				theRole = new Role();
			theRole = new Role();
			theRole.addMember(requester);
			put(roleName,theRole);
			//get(MEMBER_DEFAULT_ROLE).addMember(requester);
			return 1;
		}
	}

	int requestRole(final AgentAddress requester, final String roleName,final Object memberCard) throws CGRException
	{
		if(groupGate != null)
			if(groupGate.allowAgentToTakeRole(requester, roleName, memberCard))
				return roleAssignment(requester, roleName);
			else
				throw new CGRException(-5,null,null,roleName,new AccessDeniedException());	// access denied
		else
			return roleAssignment(requester, roleName);
	}


	/**the agent leaves the group
	 * @return true if the group is no longer useful i.e. no member, no overlooker
	 * @throws NotAMemberException if the requester agent was not is not a member of the group 
	 * 
	 */
	boolean leave(final AgentAddress requester) throws NotAMemberException 
	{
		try {
			if(get(MEMBER_DEFAULT_ROLE).removeMemberAndCheck(requester))
			{
				remove(MEMBER_DEFAULT_ROLE);
			}
		} catch (final NullPointerException e) {
			System.err.println("!!!!!! This should not happen : kernel internal error ");
			e.printStackTrace();
		} catch (RoleNotHandled e) {
			throw new NotAMemberException();
		}

		for (final Iterator<Role> i=values().iterator(); i.hasNext(); )
			if(i.next().removeMember(requester))
				i.remove();
		return super.isEmpty();
	}

	/**
	 * removing agents from a particular distant kernel or from all other kernels.
	 * 
	 * @param distantAddress the distant address. If null, the operation will remove all the agents which are from other kernels 
	 * 
	 * @return true if the group is no longer useful i.e. no member, no overlooker
	 * 
	 */
	boolean removeAgentsFromKernel(final KernelAddress distantAddress)
	{
		Collection<AgentAddress> agentsToRemove = new ArrayList<AgentAddress>(size());
		if(distantAddress==null){
			for(AgentAddress add : get(MEMBER_DEFAULT_ROLE))
				if(! add.isLocal())
					agentsToRemove.add(add);
		}
		else
			for(AgentAddress add : get(MEMBER_DEFAULT_ROLE))
				if(add.getKernel().equals(distantAddress))
					agentsToRemove.add(add);
		for(AgentAddress add : agentsToRemove)
			for (final Iterator<Role> i=values().iterator(); i.hasNext(); )
				if(i.next().removeMember(add))
					i.remove();
		return super.isEmpty();
	}


////////////il faut savoir si le fait que le group manager sorte annule l'existance du group ou translation vers un nouvel agent...
	/**
	 * Leave role.
	 * 
	 * @param requester the requester
	 * @param roleName the role name
	 * 
	 * @return true, if, after the leave operation, the group is empty.
	 * @throws CGRException 
	 */
	boolean leaveRole(final AgentAddress requester, final String roleName) throws CGRException 
	{
		try {
			if(get(roleName).removeMemberAndCheck(requester))
			{
				remove(roleName);
				return super.isEmpty();
			}
		} catch (final NullPointerException e) {
			throw(new CGRException(-3,null,null,roleName,e));
		} catch (RoleNotHandled e) {
			throw(new CGRException(-6,null,null,roleName,e));
		}
		return false;	
	}

//////////////////////////////////////////////////OVERLOOKER OPERATIONS

	boolean addOverlooker(final AgentAddress requester, final Overlooker<? extends AbstractAgent> o, final Object accessCard)
	{
		if(groupGate != null && ! groupGate.allowOverlooking(requester, accessCard))
			return false;
		if(containsKey(o.role))
			return get(o.role).addOverlooker(o);
		else
		{
			Role newRole;
			if(Kernel.agressiveHeapMode)
				newRole = new Role(Kernel.defaultAgentsAllocation);
			else
				newRole = new Role();
			put(o.role, newRole);
			return newRole.addOverlooker(o);
		}
	}

	boolean removeOverlooker(final Overlooker<? extends AbstractAgent> o){
		try {
			if (get(o.role).removeOverlooker(o))
			{
				remove(o.role);
				return super.isEmpty();
			}
		} catch (final NullPointerException e) {
		}
		return false;
	}




////////////////////////////////////////////INFO OPERATIONS
	boolean isDistributed(){
		return distributed;
	}

	/**checks if the group has no member.
@return true if there is no member. i.e. this group is a virtual one.*/
	@Override
	public boolean isEmpty() {
		try {
			return get(MEMBER_DEFAULT_ROLE).isEmpty();
		} catch (final NullPointerException e) {
		}
		return true;
	}

	boolean isPlayingRole(final AgentAddress agent,final String theRole) {
		try {
			return get(theRole).contains(agent);
		} catch (final NullPointerException e) {
			return false;
		}
	}

	Collection<AgentAddress> getRolePlayers(final String roleName) throws CGRException {
		final Collection<AgentAddress> agents = get(roleName);
		if(agents == null)
			throw new CGRException(-3,null,null,roleName,null);
		return agents;
	}

//	Collection<AgentAddress> getRolePlayers(final String roleName) {
//	return get(roleName);
//	}

	AgentAddress getRolePlayer(final String roleName) throws CGRException {
		try {
			final java.util.List <AgentAddress>  rolePlayers = new java.util.ArrayList<AgentAddress>(getRolePlayers(roleName));
			return rolePlayers.get((int) (Math.random()*rolePlayers.size()));
		} catch (NullPointerException e) {
			throw new CGRException(-3,null,null,roleName,e);
		} catch (IndexOutOfBoundsException e) {
			System.err.println("!!!!!! This should not happen : kernel internal error ");
			e.printStackTrace();
			return null;
		}
	}

	String[] getRolesOf(final AgentAddress agent)
	{
		final Collection<String> c = new TreeSet<String>();
		for (final Map.Entry<String, Role> e : entrySet()){
			if(e.getValue().contains(agent))
				c.add(e.getKey());
		}
		return c.toArray(new String[c.size()]);
	}	

	String[] availableRoles()
	{
		final Collection<String> c = new TreeSet<String>();
		for (final Map.Entry<String, Role> e : entrySet()){
			if(! e.getValue().isEmpty())
				c.add(e.getKey());
		}
		return c.toArray(new String[c.size()]);
	}

////////////////////: for Backward Compatibility : à foutre à la poubelle ...
	Map<String,Collection<Role>> mapForm()
	{
		final Map<String,Collection<Role>> rolesName = new Hashtable<String,Collection<Role>>();
		for(final Map.Entry<String, Role> entry : entrySet())
		{
			if(! entry.getValue().isEmpty())
				rolesName.put(entry.getKey(),new Vector(entry.getValue()));
		}	
		return rolesName;
	}

///////////////////////////////////////		IMPORT & ACCESSORs

	void merge(final Group g, final boolean priority)
	{
		if (priority)
		{
			try {
				if(g.getRolePlayer(GROUP_MANAGER) != null)
					realCreation(g.getRolePlayer(GROUP_MANAGER), true, g.getDescription(),g.getGroupIdentifier());
			} catch (CGRException e) {
				// TODO Auto-generated catch block
				//e.printStackTrace();
			}
		} else
			try {
				if(getRolePlayer(GROUP_MANAGER) == null || (isEmpty() && g.getRolePlayer(MEMBER_DEFAULT_ROLE) != null))
					realCreation(g.getRolePlayer(MEMBER_DEFAULT_ROLE), true, getDescription(),getGroupIdentifier());
			} catch (CGRException e) {
				// TODO Auto-generated catch block
				//e.printStackTrace();
			}
			if(isEmpty())
			{
				Role members;
				if(Kernel.agressiveHeapMode)
					members = new Role(Kernel.defaultAgentsAllocation);
				else
					members = new Role();
				put(MEMBER_DEFAULT_ROLE,members);
			}

			for(final String roleName : g.keySet())
			{
				if(! roleName.equals(GROUP_MANAGER))
				{
					for(final AgentAddress agent : g.get(roleName))
						try {
							roleAssignment(agent,roleName);
						} catch (CGRException e) {
							//System.err.println("!!!!!! This should not happen : kernel "+Kernel.getAddress()+" internal error : merge fails on group "+g +" with agent"+agent);
							//e.printStackTrace();
						}
				}
			}
	}

//	backup
	/*void merge(final Group g, final boolean priority)
{
	if (priority)
	{
		if(g.getRolePlayer("group manager") != null)
			realCreation(g.getRolePlayer("group manager"), true, g.getDescription(),g.getGroupIdentifier());
	}		
	else
		if(getRolePlayer("group manager") == null || (isEmpty() && g.getRolePlayer(MEMBER_DEFAULT_ROLE) != null))
			realCreation(g.getRolePlayer(MEMBER_DEFAULT_ROLE), true, getDescription(),getGroupIdentifier());
	if(isEmpty())
	{
		Role members;
		if(Kernel.agressiveHeapMode)
			members = new Role(Kernel.defaultAgentsAllocation);
		else
			members = new Role();
		put(MEMBER_DEFAULT_ROLE,members);
	}

	for(final String roleName : g.keySet())
	{
		if(! roleName.equals("group manager"))
		{
			for(final AgentAddress agent : g.get(roleName))
				roleAssignment(agent,roleName);
		}
	}
}
	 */
	GroupIdentifier getGroupIdentifier(){return groupGate;}
	String getDescription(){return description;}

	void removeAgent(Kernel kernel, AgentAddress theAgent, String orgName, String groupName) {
		if (get(MEMBER_DEFAULT_ROLE).contains(theAgent)){
			try {
				kernel.leaveGroup(theAgent,orgName,groupName);
			} catch (LeaveGroupException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}

	}

	boolean removeAgentFromGroup(AgentAddress theAgent) {
		boolean in = false;
		for (Iterator<Role> i = values().iterator();i.hasNext();) {
			Role r = i.next();
			if(r.removeMember(theAgent)){
				if(r.isEmpty())
					i.remove();
				in = true;
			}
		}
		return in;			

	}

	@Override
	public void clear() {
		for (Role r : values()) {
			r.clear();	
		}
		super.clear();
	}
}