/*
 * Agent.java - Kernel: the kernel of MadKit
 * Copyright (C) 1998-2008 Olivier Gutknecht, Fabien Michel, Jacques Ferber
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package madkit.kernel;

/** The main MadKit AbstractAgent class. Provides support for non threaded agent
 * life cycle, messaging, graphical interface, group and roles
 * management, agent information, ...
 *<p>
 * The agent behavior is intentionally *not* defined. It is up to the
 * agent developer to choose an agent model or to develop his specific
 * agent library on top of the facilities provided by MadKit. However,
 * all agent share the same organizational view, and the basic
 * messaging code, so integration of agent coming from different
 * developer and having heterogeneous models is quite easy.
 *<p>
 * Agent-related methods (almost everything here) can invoked only
 * after registration in the kernel (i.e. starting from the activate() method call
 * on this agent). That means that you should not use any of the agent methods in
 * constructor
 *
  @author Olivier Gutknecht
  @author Fabien Michel : MadKit 3.0 revision
  @version 4.3
 */

public abstract class Agent extends AbstractAgent implements Runnable, java.io.Serializable
{
	transient Thread agentThread = new Thread(); // a fake object to type distinguish threaded agents from others

	//////////////////////////////////////////////
	// ------ Thread managment
	/** Suspend the agent for a while. The main agent thread is put to
      sleep for the specified duration. If the agent developer creates
      additional threads, they will not be suspended
      @param t pause duration in milliseconds */

	final public void pause(int t)
	{
		try {Thread.sleep(t);}
		catch (InterruptedException e) {}
	}

	/**
	 * @return the agent thread
	 */
	final Thread getAgentThread()
	{
		return agentThread;
	}

	/////////////////////////////////////////////
	// ------ Messaging interface

	/**
	 * This method is the blocking version of nextMessage().  It
	 * suspends the main agent thread (but the children of the agent)
	 * until a message has been received
	 * 
	 * @return the message
	 */
	final public Message waitNextMessage()
	{
		try
		{
			synchronized(messageBox)
			{
				while(messageBox.isEmpty())
				{
					messageBox.wait();
				}
			}
			return nextMessage();
		}
		catch (InterruptedException e)
		{
			return null;
		}
	}

	/**
	 * This method is the blocking version of nextMessage(), with
	 * time-out.  It suspends the main agent thread until a message has
	 * been received or the time out delay elapsed (and returns null in
	 * that case)
	 * 
	 * @param timeout the maximum time to wait, in milliseconds
	 * 
	 * @return the message
	 */
	final public Message waitNextMessage(long timeout)
	{
		try
		{
			synchronized(messageBox)
			{
				if(messageBox.isEmpty())
					messageBox.wait(timeout);
			}
			return nextMessage();
		}
		catch (InterruptedException e)
		{
			return  null;
		}
	}

	/** this method is dramatically private. Nyark ;-) */
	@Override
	protected void receiveMessage(Message m)
	{
		super.receiveMessage(m);
		synchronized (messageBox)
		{
			messageBox.notifyAll();
		}
	}

	/** This method defines the main behavior of threaded agents. */
	public void live() {}

	/** Only useful to the agent micro-kernel */
	final public void run()
	{
		try
		{
			agentThread = Thread.currentThread();
			Controller c = getController();
			// if there is a controller, the controller takes over
			if (c != null)
			{
				c.activate();
				c.live();
			}
			else // else standard behavior..
			{
				activate();
				live();
			}
		}
//		catch (InterruptedException e) {
//			System.err.println("\ninterrupted");
//		}
		catch (RuntimeException r)
		{
			if(currentKernel == null)
				return;
			currentKernel.displayln ("Runtime Agent Exception : "+this.getName()+" killed !");
			r.printStackTrace();
		}
		catch (NoClassDefFoundError e)	{
			System.err.println("Exception class not found :::::::::::::::::::::::::::::"+e.toString());
		}
		catch (Exception e)
		{
			if(currentKernel == null)	return;
			currentKernel.display("### UNDEFINED EXCEPTION ! : " + e.toString());
			e.printStackTrace();
			currentKernel.display(this+"Agent killed. (MadKit internal error)");
			currentKernel.display("Please send a bug report !");
		}
		if(currentKernel != null)
			currentKernel.killAgent(this);
	}
//	added by saber
//	A modifier ces deux méthodes avec d'autres qui traitent avec le place keeper.

	public void createPlace(String place,String informations)
	{
		getCurrentKernel().createPlace(getAddress(),place,informations);
	}
	public void createPlace(String place)
	{
		createPlace(place,"");
	}
	public void enableMobility(String name) // this method make the kernel a mobile agency it will result that an agencyKeepr
	{
		enableMobility(name,4444);
	}

	public void enableMobility(String name, int port) // this method make the kernel a mobile agency it will result that an agencyKeepr
	{
		getCurrentKernel().enableMobility(name,port);
		// We must add this waiting time to avoid problems.
		//Problems can happen if we need the mobility group just after launching the agencyKeeper
		// and the kernel launches it after few millisecnds so not really in real time.
		pause(500);
	}
	public boolean connectAgencyToAgency(String host, int port) // this method make the kernel a mobile agency it will result that an agencyKeepr
	{
		if(getCurrentKernel().connectAgencyToAgency(host,port))
			return true;
		else
		{ 
			System.err.println("You must enable mobility before requesting connexion to agency!");
			return false;
		}	
	}		

	public int joinPlace(String place)
	{
		return (joinPlace(place,""));
	}
	public int joinPlace(String Community, String place,String pwd)// in public Community
	{
		pause(500); //  Unless for the first time the creator of the place won't be registred!
		if(getCurrentKernel().joinPlace(this,Community,place,pwd)==1);
		{
			joinGroup(place);
		}
		return (getCurrentKernel().joinPlace(this,Community,place,pwd));
	}

	public int joinPlace(String place,String pwd) // In a specific community
	{
		return(joinPlace("public",place,pwd));
	}


	public KernelAddress getAgencyNamed(String name)
	{
		return getCurrentKernel().getAgencyNamed(name);
	}

//	end of saber contribution

/** Ensures that an agent will be cleanly killed.
 * Due to the "Thread problem" (see  <a href="http://java.sun.com/j2se/1.5.0/docs/guide/misc/threadPrimitiveDeprecation.html"> Why are Thread.stop, Thread.suspend and Thread.resume Deprecated?.</a> for more details)
*when an agent is killed, by another agent or interface (i.e. without exiting of the <code>live</code> method)
*the Kernel will try a Thread.stop on it. And the Thread problem is that this method is currently unsafe and does not work
*every time. 
*<p>
*The result is that the agent has been killed, that is to say removed from organizations and invisible to others (no more messages),
*but its thread is still alive with unpredictable behaviors, wasting cpu time.
*<p>
*The sun API says :
*<i>Most uses of stop should be replaced by code that simply modifies some variable to indicate that the target thread should stop running. The target thread
*should check this variable regularly, and return from its run method in an orderly fashion if the variable indicates that it is to stop running. (This is the approach
*that JavaSoft's Tutorial has always recommended.)</i>
*<p>
*
*In MadKit, agents are autonomous :)
*
*<p>
*So it is the developer responsibility to be sure that an agent cleanly exits its thread when killed.
*<p>
*The <code>exitImmediatlyOnKill</code> method is made to ease this procedure.
*<p>
*When a dead agent, which has been killed, calls this method, its thread immediately exits if it is still running, avoiding unpredictable behavior
*<p>
*For example this kind of agent may not be able to be cleanly killed by another due to its simplicity (no wait, no pause).
* <p><hr><blockquote><pre>
* public void live() {
	int i=0;
	while(true)	//UNSAFE
		i++;
}
* </pre></blockquote><hr>
*instead of that, use the following :
* <p><hr><blockquote><pre>
public void live() {
	int i=0;
	while(true)	{
		exitImmediatlyOnKill();		//SAFE
		i++;
	}
}
* </pre></blockquote><hr>
*/

//	Makes the thread crash itself
	final protected void exitImmediatlyOnKill()
	{
		getMyGroups();
	}


}

