/*
 * InstallThread.java
 * Copyright (C) 1999, 2000 Slava Pestov
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package installer;

import java.io.*;
import java.util.Vector;
import java.util.zip.*;

/*
 * The thread that performs installation.
 */
public class InstallThread extends Thread
{
	public InstallThread(Install installer, Progress progress,
		String installDir, String binDir, int size, Vector components)
	{
		super("Install thread");

		this.installer = installer;
		this.progress = progress;
		this.installDir = installDir;
		this.binDir = binDir;
		this.size = size;
		this.components = components;

		buf = new byte[32768];
	}

	public void setProgress(Progress progress)
	{
		this.progress = progress;
	}

	public void run()
	{
		progress.setMaximum(components.size());

		try
		{  
			// create it in case it doesn't already exist
			
			if(installDir != null)
				OperatingSystem.getOperatingSystem().mkdirs(installDir);
			
			for(int i = 0; i < components.size(); i++)
			{ PluginInfo plug = (PluginInfo) components.get(i);
			  if (plug.isInstallIt())
				installComponent(plug.getPluginName()+"-"+plug.getPluginVersion());
			}


			OperatingSystem.getOperatingSystem().createScript(
				installer,installDir,binDir,
				installer.getProperty("app.name"));
		}
		catch(FileNotFoundException fnf)
		{
			progress.error("The installer could not create the "
				+ "destination directory.\n"
				+ "Maybe you do not have write permission?");
			return;
		}
		catch(IOException io)
		{
			progress.error(io.toString());
			return;
		}

		progress.done();
	}

	// private members
	private Install installer;
	private Progress progress;
	private String installDir;
	private String binDir;
	private int size;
	private Vector components;
	private byte[] buf;

//	private void installComponent(String name) throws IOException
//	{
//		BufferedReader fileList = new BufferedReader(
//			new InputStreamReader(getClass()
//			.getResourceAsStream(name)));
//
//		String fileName;
//		while((fileName = fileList.readLine()) != null)
//		{
//			String outfile = installDir + File.separatorChar
//				+ fileName.replace('/',File.separatorChar);
//
//			InputStream in = new BufferedInputStream(
//				getClass().getResourceAsStream("/" + fileName));
//
//			if(in == null)
//				throw new FileNotFoundException(fileName);
//
//			copy(in,outfile);
//			in.close();
//		}
//
//		fileList.close();
//	}
	
	private void installComponent(String name) throws IOException
		{		
			System.out.println("== trying to install: "+name);
			String filename = name+".zip";

			InputStream in = new BufferedInputStream(
				getClass().getResourceAsStream(filename));

			if(in == null)
				throw new FileNotFoundException(filename);
			System.out.println("extracting..."+filename+" into "+installDir);
			extract(in);
			in.close();

			progress.advance(1);
		}

	private void copy(InputStream in, String outfile) throws IOException
	{
		File outFile = new File(outfile);

		OperatingSystem.getOperatingSystem().mkdirs(outFile.getParent());

		BufferedOutputStream out = new BufferedOutputStream(
			new FileOutputStream(outFile));

		int count;

		for(;;)
		{
			count = in.read(buf,0,buf.length);
			if(count == -1)
				break;

			out.write(buf,0,count);
			progress.advance(count);
		}

		in.close();
		out.close();
	}
	
	boolean debug=false;
	
	public void extract(InputStream zipFile){
			ZipInputStream zis=null;
			int num_entries=0;

			byte [] buffer = new byte[4096];
			long total_bytes=0;
//		
//			Enumeration entries = zipFile.entries();

//			while ( entries.hasMoreElements() ) 
			try {
				ZipEntry zipEntry;
				zis = new ZipInputStream(zipFile);
				do {
					zipEntry = zis.getNextEntry();
					//ZipEntry zipEntry = (ZipEntry)entries.nextElement();
					// zis = new ZipInputStream(zipFile.getInputStream(zipEntry));
					if (zipEntry != null ) {
							//println( "entry:"+zipEntry.getName() );
							++num_entries;
							int method = zipEntry.getMethod();
							if (debug) {
								System.out.println( "*** ENTRY ["+num_entries+"] ***" );
								System.out.println( "    name: " + zipEntry.getName() );
								System.out.println( "    size: " + zipEntry.getSize() );
								System.out.println( "    extra: " + zipEntry.getExtra() );
								System.out.println( "    compressed size: " + zipEntry.getCompressedSize() );
								System.out.println( "    method: " +
											(method == ZipEntry.DEFLATED ? "(Compressed)" :
											 method == ZipEntry.STORED ? "(Stored)" : "UNKNOWN!" ));
							}
	
							String entryFilePath = zipEntry.getName();
							entryFilePath = entryFilePath.replace('/', File.separatorChar );
							entryFilePath = entryFilePath.replace('\\', File.separatorChar );
							//System.out.println( "entry extract: `"+entryFilePath +"'");
			
			
							File entryFile = new File(installDir, entryFilePath );
							if (zipEntry.isDirectory() ) {
								// This entry is a directory, try to create it.
								entryFile.mkdirs();
							} else {
								// Extract the zip file entry
								if ( entryFile.getParent() != null )
									new File(entryFile.getParent()).mkdirs();
				    
								FileOutputStream fos = new FileOutputStream( entryFile );
								int num_bytes=0;
								while ((num_bytes = zis.read(buffer, 0, buffer.length )) >= 0 ) {
									fos.write( buffer, 0, num_bytes );
									fos.flush();	// FLUSH IT!
									total_bytes += num_bytes;
								} 
//								int c;
//								while ((c = zis.read()) != -1){
//									fos.write(c);
//									num_bytes++;
//								}
								//println("bytes: "+total_bytes);
								fos.close();		// CLOSE IT!!
							}
							zis.closeEntry();
						}
					Thread.yield();
				}  while (zipEntry != null);

				System.out.println("Number of entries : "+num_entries);
			}
				catch(IOException e){
					System.out.println("Error in extracting file "+ zipFile);
					System.out.println(e.getMessage());
					e.printStackTrace();
				}
				finally {
					// Use the finally clause to make sure the stream is closed.
					if (zis != null) try { zis.close(); } catch (IOException e ) { ; }
				}
		}
}
