
/*
* Genetic Package for TurtleKit 2: TurtleKit - A 'reactive simulation platform' using MadKit Kernel
* Copyright (C) 2002-2007 Gregory Beurier
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/


package turtlekit2.genetic;

import java.awt.Component;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;

import javax.swing.JFileChooser;
import javax.swing.filechooser.FileFilter;



/**
 * <p>Title : GeneticFileManager </p>
 * <p>Description: IO Class for genetic files. Load and Save Genes, Genomes and Pools <p>
 * <p>Standard extensions are .gene, .gnom, .pool . Saved genetics files are serializable objects. 
 * XML saving has been disabled </p>
 * @author Gregory Beurier
 */  

public class GeneticFileManager {
	
	public final static String pool = "pool";

	public final static String gnom = "gnom";

	public final static String gene = "gene";
	
	
	
	
	static public void savePool(Component f, Pool o) {
		save(f,pool,(Object)o);
	}
	static public void saveGenome(Component f, Genome o) {
		save(f,gnom,(Object)o);
	}
	static public void saveGene(Component f, Gene o) {
		save(f,gene,(Object)o);
	}
	
	
	static public void savePool(String path, Pool o) {
		save(path,pool,(Object)o);
	}
	static public void saveGenome(String path, Genome o) {
		save(path,gnom,(Object)o);
	}
	static public void saveGene(String path, Gene o) {
		save(path,gene,(Object)o);
	}
	
	static public void save(Component f, String type, Object o) {
		JFileChooser jChooser = new JFileChooser(System.getProperty("user.dir"));
        GenFilter filter = new GenFilter(type);
        jChooser.setFileFilter(filter);
		int returnVal = jChooser.showSaveDialog(f);
		if (returnVal == JFileChooser.APPROVE_OPTION) {
			System.out.println("You chose to save this file: "+ jChooser.getSelectedFile().getAbsolutePath());
			save(jChooser.getSelectedFile().getAbsolutePath(), type, o);
		}
	}
	
	
	static public void save(String path, String type, Object o) {
		String fileName = path + "." + type;
		try {
			File poolFile = new File(fileName);
			FileOutputStream fos = new FileOutputStream(poolFile);
			ObjectOutputStream oos = new ObjectOutputStream(fos);
			oos.writeObject(o);
			oos.flush();
			oos.close();
		} catch (IOException e) {
			System.err.println(type + " File Write Error !\n" + fileName + "\n" + e);
		}
	}
	
	
	
	
	static public Pool loadPool(Component f){
		return (Pool)load(f,pool);
	}
	
	static public Genome loadGenome(Component f){
		return (Genome)load(f,gnom);
	}
	
	static public Gene loadGene(Component f){
		return (Gene)load(f,gene);
	}
	
	static public Object load(Component f, String type) {
        JFileChooser jChooser = new JFileChooser(System.getProperty("user.dir"));
        GenFilter filter = new GenFilter(type);
        jChooser.setFileFilter(filter);
        int returnVal = jChooser.showOpenDialog(f);
        if(returnVal == JFileChooser.APPROVE_OPTION) {
            System.out.println("You chose to open this file: " +jChooser.getSelectedFile().getAbsolutePath());
            return load(jChooser.getSelectedFile().getAbsolutePath());
        }else return null;
        
    }

	static public Object load(String path) {
		String fileName = path;
		try {
			File poolFile = new File(fileName);
			FileInputStream fis = new FileInputStream(poolFile);
			ObjectInputStream ois = new ObjectInputStream(fis);
			Object loadedObject = ois.readObject();
			ois.close();
			return loadedObject;
		} catch (IOException e) {
			System.err.println("File Load Error !\n" + fileName + "\n"
					+ e);
		} catch (ClassNotFoundException e) {
			System.err.println("Object ClassNotFoundError Read Error !\n"
					+ fileName + "\n" + e);
		}
		return null;
	}

	
}

/**
 * <p>Title : GeneticFileManager </p>
 * <p>Description: File type filter for genetic File manager </p>
 * @author Gregory Beurier
 */  
class GenFilter extends FileFilter {
	public final static String pool = "pool";

	public final static String gnom = "gnom";

	public final static String gene = "gene";
	
	public String type = "pool";

	public GenFilter(String type) {
		super();
		this.type = type;
	}

	public boolean accept(File f) {
		if (f.isDirectory()) {
			return true;
		}
		String extension = getExtension(f);
		if (extension != null) {
			if (extension.equals(type)) {
				return true;
			} else {
				return false;
			}
		}

		return false;
	}

	public String getExtension(File f) {
		String ext = null;
		String s = f.getName();
		int i = s.lastIndexOf('.');

		if (i > 0 && i < s.length() - 1) {
			ext = s.substring(i + 1).toLowerCase();
		}
		return ext;
	}

	//The description of this filter
	public String getDescription() {
		if(type.equals(pool)) return "Tk3 pool files";
		else if(type.equals(gnom)) return "Tk3 genome files";
		else if(type.equals(gene)) return "Tk3 gene files";
		return "tk3 files";
	}
}