
package modulecoGUI;

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import java.net.URL;
import java.util.Iterator;
import java.util.Map;
import java.util.TreeMap;

import javax.swing.ButtonGroup;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.JRootPane;
import javax.swing.JSeparator;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;

import modulecoFramework.Moduleco;
import modulecoGUI.XMLGuiReader.XMLGuiReader;
import modulecoGUI.debug.JFrameDebug;

/**
 * @author Thibaud Roussillat
 * @version 1.5.1a : last modify on 20 September 2005
 *
 * <br>Cette classe reprsente le container de plus haut niveau de l'interface.
 * Le fenetrage est lui gr par Madkit qui s'occupe de mettre
 * ce JRootPane dans une fentre<br>
 */

public class ModulecoBean extends JRootPane implements ActionListener {

	/** serialVersionUID */
	private static final long serialVersionUID = 809409513531068575L;
	
	/** Le panneau principal de l'application */
	private JPanelMain jPanelMain;
	
	/** Le contentPane du JRootPane */
	private JPanel contentPane;
	
	/** Le gestionnaire d'interaction de l'interface */
	private CentralControl centralControl;
	
	/** La bar de menu de l'interface **/
	private JMenuBar jMenuBar;
	
	/** Le JMenu File et ses JMenuItem */
	private JMenu jMenuFile;
	private JMenuItem jMenuItemOpen;
	private JMenuItem jMenuItemOpenRecentFiles;
	private JMenuItem jMenuItemClose;
	
	/** Le JMenu Edit et ses JMenuItem */
	private JMenu jMenuEdit;
	private JMenuItem jMenuItemCut;
	private JMenuItem jMenuItemCopy;
	private JMenuItem jMenuItemPaste;
	
	/** Le JMenu Display et ses JMenuItem */
	private JMenu jMenuDisplay;
	private JMenuItem jMenuItemDockingHierarchy;
	private JMenuItem jMenuItemModelProperties;
	private JMenuItem jMenuItemNeighbourhood;
	
	/** Le JMenu Tools et ses JMenuItem */
	private JMenu jMenuTools;
	private JMenuItem jMenuItemLanguage;
	
	/** Le JMenu Window et ses JMenuItem */
	private JMenu jMenuWindow;
	private JMenuItem jMenuItemResetPerspective;
	private JMenuItem jMenuShowView;
	
	/** Le JMenu Madkit Tools et ses JMenuItem */
	private JMenu jMenuMadkitTools;
	private JMenuItem jMenuItemExplorer;
	private JMenuItem jMenuItemGroupObserver;
	private JMenuItem jMenuItemOutput;
	
	/** Le JMenu Data et ses JMenuItem */
	private JMenu jMenuData;
	private JMenuItem jMenuItemRecordData;

	/** Le JMenu Look and Feel et ses JMenuItem */
	private JMenu jMenuLookAndFeel;
	
	/** Le JMenu Help et ses JMenuItem */
	private JMenu jMenuHelp;
	private JMenuItem jMenuItemAboutModuleco;
	
	/**
	 * Construit un nouveau ModulecoBean
	 * @param centralControl : le gestionnaire d'interaction de l'interface graphique
	 */
	public ModulecoBean(CentralControl centralControl) {
		
		/* Affectation du centralControl */
		this.centralControl = centralControl;
		
		/* configuration de la taille par dfaut de la fentre */
		this.setPreferredSize(new Dimension(800,600));
		
		/* Recupration du contentPane */
		this.contentPane = (JPanel)this.getContentPane();
		
		/* Configuration du Layout du contentPane */
		this.contentPane.setLayout(new BorderLayout());
		
		/* Appel de la mthode qui sert  construire le menu */
		this.buildMenu();
		
		/* Appel de la mthode qui sert  construire l'interface */
		this.buildInterface();
		
	}
	
	/**
	 * Effectue les oprations d'initialisation qui ne peuvent pas tre faites auparavant :<br>
	 * Agrandissement de la fentre et chargement de l'icone
	 */
	public void init() {
		/* Rcupration du container de plus haut niveau ( =fentre) */
		Container cont = SwingUtilities.getAncestorOfClass(JFrame.class, this);
		/* Castage en JFrame */
        JFrame jFrame = (JFrame) cont;
        /* Extension de la fentre  sa taille maximum */
        jFrame.setExtendedState(JFrame.MAXIMIZED_BOTH);
        /* Rcupration de l'image et affichage en tant qu'icone */
        URL url = this.getClass().getResource("/images/logo-small.png");
		ImageIcon icon = new ImageIcon(url,"logo");
        jFrame.setIconImage(icon.getImage());
	}
	
	
	/**
	 * Construit ce qu'il y a de contenu  l'intrieur de ModulecoBean, 
	 * c'est  dire le JPanelMain contenant la Barre d'outil, le panneau du milieu
	 * et la barre d'tat.
	 */
	private void buildInterface() {
		
		/* Cration d'un nouveau JPanelMain */
		this.jPanelMain = new JPanelMain(this.centralControl,this);
		/* Ajout du JPanelMain au contentPane */
		this.contentPane.add(jPanelMain,BorderLayout.CENTER);
		
	}
	
	
	/** 
	 * Construit le menu de la fentre.
	 * Les items du menu sont en gnral statiques mais leur label est charg  partir du fichier xml
	 * de l'interface appel GUI.language.xml (par exemple GUI.franais.xml ou GUI.english.xml). Ce chargement 
	 * s'effectue  travers XMLGuiReader qui lie le fichier xml.
	 * Certains menu sont construits dynamiquements en rcuprant des informations via des appels de mthode.
	 */
	private void buildMenu() {
		
		/* Construction d'une nouvelle barre de menu */
		jMenuBar = new JMenuBar();
		
		/* Construction du menu File et de ses items */
		jMenuFile = new JMenu(XMLGuiReader.getLabel("ModulecoBean/MenuFile"));
		jMenuItemOpen = new JMenuItem(XMLGuiReader.getLabel("ModulecoBean/MenuItemOpen"));
		jMenuItemOpenRecentFiles  = new JMenuItem(XMLGuiReader.getLabel("ModulecoBean/MenuItemOpenRecentFiles"));
		jMenuItemClose = new JMenuItem(XMLGuiReader.getLabel("ModulecoBean/MenuItemClose"));
		
		jMenuItemOpen.addActionListener(this);
		jMenuItemOpenRecentFiles.addActionListener(this);
		jMenuItemClose.addActionListener(this);
		
		jMenuFile.add(jMenuItemOpen);
		jMenuFile.add(jMenuItemOpenRecentFiles);
		jMenuFile.add(new JSeparator());
		jMenuFile.add(jMenuItemClose);
		
		/* Construction du menu Edit et de ses items */
		jMenuEdit = new JMenu(XMLGuiReader.getLabel("ModulecoBean/MenuEdit"));
		jMenuItemCut = new JMenuItem(XMLGuiReader.getLabel("ModulecoBean/MenuItemCut"));
		jMenuItemCopy = new JMenuItem(XMLGuiReader.getLabel("ModulecoBean/MenuItemCopy"));
		jMenuItemPaste = new JMenuItem(XMLGuiReader.getLabel("ModulecoBean/MenuItemPaste"));
		
		jMenuItemCut.addActionListener(this);
		jMenuItemCopy.addActionListener(this);
		jMenuItemPaste.addActionListener(this);
		
		jMenuEdit.add(jMenuItemCut);
		jMenuEdit.add(jMenuItemCopy);
		jMenuEdit.add(jMenuItemPaste);
		
		/* Construction du menu Display et de ses items */
		jMenuDisplay = new JMenu(XMLGuiReader.getLabel("ModulecoBean/MenuDisplay"));
		jMenuItemDockingHierarchy = new JMenuItem(XMLGuiReader.getLabel("ModulecoBean/MenuItemDockingHierarchy"));
		jMenuItemModelProperties = new JMenuItem(XMLGuiReader.getLabel("ModulecoBean/MenuItemModelProperties"));
		jMenuItemNeighbourhood = new JMenuItem(XMLGuiReader.getLabel("ModulecoBean/MenuItemNeighbourhood"));
		
		jMenuItemDockingHierarchy.addActionListener(this);
		jMenuItemModelProperties.addActionListener(this);
		jMenuItemNeighbourhood.addActionListener(this);
		
		jMenuDisplay.add(jMenuItemDockingHierarchy);
		jMenuDisplay.add(jMenuItemModelProperties);
		jMenuDisplay.add(jMenuItemNeighbourhood);
		
		/* Construction du menu Tools et de ses items */
		jMenuTools = new JMenu(XMLGuiReader.getLabel("ModulecoBean/MenuTools"));
		jMenuItemLanguage = new JMenuItem(XMLGuiReader.getLabel("ModulecoBean/MenuItemLanguage"));
		/* Construction du menu de choix de langue */
		this.buildMenuLanguage(jMenuItemLanguage);
		
		jMenuItemLanguage.addActionListener(this);
		
		jMenuTools.add(jMenuItemLanguage);
		
		/* Construction du menu Window et de ses items */
		jMenuWindow = new JMenu(XMLGuiReader.getLabel("ModulecoBean/MenuWindow"));
		jMenuItemResetPerspective = new JMenuItem(XMLGuiReader.getLabel("ModulecoBean/MenuItemResetPerspective"));
		jMenuShowView = new JMenu(XMLGuiReader.getLabel("ModulecoBean/MenuShowView"));
		
		jMenuItemResetPerspective.addActionListener(this);
		jMenuShowView.addActionListener(this);
		
		jMenuWindow.add(jMenuItemResetPerspective);
		jMenuWindow.add(jMenuShowView);
		
		/* Construction du menu Madkit Tools et de ses items */
		jMenuMadkitTools = new JMenu(XMLGuiReader.getLabel("ModulecoBean/MenuMadkitTools"));
		jMenuItemExplorer = new JMenuItem(XMLGuiReader.getLabel("ModulecoBean/MenuItemExplorer"));
		jMenuItemGroupObserver = new JMenuItem(XMLGuiReader.getLabel("ModulecoBean/MenuItemGroupObserver"));
		jMenuItemOutput = new JMenuItem(XMLGuiReader.getLabel("ModulecoBean/MenuItemOutput"));
		
		jMenuItemExplorer.addActionListener(this);
		jMenuItemGroupObserver.addActionListener(this);
		jMenuItemOutput.addActionListener(this);
		
		jMenuMadkitTools.add(jMenuItemExplorer);
		jMenuMadkitTools.add(jMenuItemGroupObserver);
		jMenuMadkitTools.add(jMenuItemOutput);
		
		/* Construction du menu Data et de ses items */
		jMenuData = new JMenu(XMLGuiReader.getLabel("ModulecoBean/MenuData"));
		jMenuItemRecordData = new JMenuItem(XMLGuiReader.getLabel("ModulecoBean/MenuItemRecordData"));
		
		jMenuItemRecordData.addActionListener(this);
		
		jMenuData.add(jMenuItemRecordData);
		
		jMenuLookAndFeel = new JMenu(XMLGuiReader.getLabel("ModulecoBean/MenuLookAndFeel"));
		this.buildMenuLookAndFeel();
		
		/* Construction du menu Help et de ses items */
		jMenuHelp = new JMenu(XMLGuiReader.getLabel("ModulecoBean/MenuHelp"));
		jMenuItemAboutModuleco = new JMenuItem(XMLGuiReader.getLabel("ModulecoBean/MenuItemAboutModuleco"));
		
		jMenuHelp.add(jMenuItemAboutModuleco);
		
		/* Ajout des menus  la bar de menu */
		jMenuBar.add(jMenuFile);
		jMenuBar.add(jMenuEdit);
		jMenuBar.add(jMenuDisplay);
		jMenuBar.add(jMenuTools);
		jMenuBar.add(jMenuWindow);
		jMenuBar.add(jMenuMadkitTools);
		jMenuBar.add(jMenuData);
		jMenuBar.add(jMenuLookAndFeel);
		jMenuBar.add(jMenuHelp);
		
		/* Ajout de la barre de menu au JRootPane */
		this.setJMenuBar(jMenuBar);
		
	}
	
	/**
	 * Construit les items avec les diffrentes langues possible
	 * Pour cela, regarde dans le rprtoire de base de Moduleco les fichiers appels
	 * GUI.langue.xml (GUI.english.xml ou GUI.franais.xml) et fait une liste des langues possibles.
	 * @param menuItemLanguage
	 */
	private void buildMenuLanguage(JMenuItem menuItemLanguage) {
		
		/** TODO : dfinir la mthode buildMenuLanguage **/
		
	}

	/**
	 * Construits le menu Look and Feel proposant ainsi les look and feel disponibles et
	 * permettant  l'utilisateur d'appliquer celui qu'il dsire.
	 * @see <a href="http://java.developpez.com/faq/java/?page=generalitesAWTSwing#lf">http://java.developpez.com/faq/java/?page=generalitesAWTSwing#lf</a>
	 * @see <a href="http://java.developpez.com/faq/java/?page=generalitesAWTSwing#listerLF">http://java.developpez.com/faq/java/?page=generalitesAWTSwing#listerLF</a>
	 */
	private void buildMenuLookAndFeel() {
		
		/* Cration d'un ButtonGroup car seulement un des item peut tre activ  la fois */
		ButtonGroup buttonGroup = new ButtonGroup();
		
		/* Rcupration des Look and Feel disponible sous forme de Map (nom : nom de la classe L&F)*/
		Map map = this.getLookAndFeelsMap();
		
		/* Itration sur les cls de la map */
		for(Iterator i = map.keySet().iterator();i.hasNext();) {
			
			/* Rcupration de la cl courante */
			String clef = (String)i.next();
			/* Rcupration du nom de la classe associ */
			final String classe = (String) map.get(clef);
			/* Si c'est le Look & Feel courant : natif est gal  true et le JRadioButton est selectionn */
			boolean natif = classe.equals(UIManager.getSystemLookAndFeelClassName());
			/* Construction du JRadioButton (label, selection {true,false}) */
			JRadioButtonMenuItem item = new JRadioButtonMenuItem(clef,natif);
			
			/* Dfinition de l'actionListener */
			item.addActionListener(new ActionListener(){ 
				public void actionPerformed(ActionEvent ae){ 
					try{ 
						/* Configuration nouveau du Look & Feel */
						UIManager.setLookAndFeel(classe); 
					}catch(Exception e){
						e.printStackTrace();	
					} 
					/* Puis force chaque composant de la fentre  appeler sa mthode updateUI */
					SwingUtilities.updateComponentTreeUI(ModulecoBean.this); 
					
				} 
			}); 
			
			/* Ajout de l'item au buttonGroup */
			buttonGroup.add(item); 
			/* Ajout de l'item au jMenu */
			jMenuLookAndFeel.add(item);  
			
		}
	}
	
	/** 
	 * Rcupre les look and feels disponibles et les retourne sous forme d'une map
	 * @return une Map associant les noms des L&F et les classe correspondantes
	 */
	private Map getLookAndFeelsMap(){
		
		/* Rcupration d'un tableau contenant les L&F disponibles */
		UIManager.LookAndFeelInfo[] info = UIManager.getInstalledLookAndFeels();
		/* Construction d'une nouvelle Map */
		Map map = new TreeMap();
		
		/* Remplissage de la map */
		for(int i=0; i<info.length; i++){
			/* Rcupration du nom du L&F */
			String nomLF = info[i].getName();
			/* Rcupration de la classe du L&F */
			String nomClasse = info[i].getClassName();
			/* Association au sein de la map */
			map.put(nomLF,nomClasse); 	
		}
		
		/* Renvoie la map remplie */
		return map;	
	}

	
	/**
	 * Retourne le jPanelMain, container du centre de l'interface (bar d'outil, centre, 
	 * et barre d'tat)
	 * @return Le JPanelMain
	 */
	public JPanelMain getJPanelMain() {
		/* Retourne le jPanelMain */
		return jPanelMain;
	}

	/* (non-Javadoc)
	 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
	 */
	/**
	 * Capte les vnements des menus et les redirigent vers le centralControl pour appeler 
	 * les mthodes correspondantes.
	 */
	public void actionPerformed(ActionEvent evt) {
		
		String actionCommand = evt.getActionCommand();
		
		/* Rcupration de l'objet source de l'vnement */
		Object source = evt.getSource();
		
		/* Debugging mode : affiche la hierarchie de Docking Panel du JPanelMain
		 * c'est  dire l'organisation des Split Window et des Onglets */
		if (source.equals(jMenuItemDockingHierarchy)) {
			centralControl.displayDockingHierarchy();
		}
		
		/* Reset perspective : rorganise la vue comme elle tait  l'initialisation */
		else if (source.equals(jMenuItemResetPerspective)) {
			centralControl.resetPerspective();
		}
		
		/* Debugging mode : affiche les propriets du modles */
		else if (source.equals(jMenuItemModelProperties)) {
			JFrameDebug jFrameDebug = new JFrameDebug();
			jFrameDebug.showModelProperties(centralControl.modulecoLauncher);
		}
		
		/* Permet d'ouvrir un nouveau modle */
		else if (source.equals(jMenuItemOpen)) {
			centralControl.openJFileChooserForModele();
		}
		
		/* Debugging mode : affiche le voisinage de chaque agent dans la console */
		else if (source.equals(jMenuItemNeighbourhood)) {
			centralControl.displayNeighbourhood();
		}
		
		/* Permet d'ouvrir le JconfigRecorder qui permet de choisir quelles donnes du modle enregistrer
		 * dans le fichier csv */
		else if (source.equals(jMenuItemRecordData)) {
			centralControl.showJConfigRecorder();
		}
		
		/* Permet d'ouvrir l'explorer Madkit */
		else if (source.equals(jMenuItemExplorer)) {
			centralControl.launchMadkitAgent("madkit.explorer.ExplorerLauncher");
		}
		
		/* Permet d'ouvrir la console Madkit */
		else if (source.equals(jMenuItemOutput)) {
			centralControl.launchMadkitAgent("madkit.output.MadkitOutput");
		}
		
		/* Permet d'ouvrir le "GroupObserver" de Madkit */
		else if (source.equals(jMenuItemGroupObserver)) {
			centralControl.launchMadkitAgent("madkit.system.GroupObserver");
		}
		
		/* Permet de mettre au premier plan la vue choisie */
		else if ("showView".equals(actionCommand.substring(0,actionCommand.indexOf(".")))) {
			/* Rcupration du nom de la vue (showView.nomVue) */
			String name = actionCommand.substring(actionCommand.indexOf(".")+1,actionCommand.length());
			centralControl.showView(name);
		}
		
		
	}

	/**
	 * Met  jour le menu "Show View" lorsqu'une nouvelle vue est cre
	 * @param name : le nom de la vue
	 * @param icon : l'icone de la vue
	 */
	public void updateMenuShowView(String name, Icon icon) {
		
		/* Cration du nouveau JMenuItem */
		JMenuItem jMenuItem = new JMenuItem(name,icon);
		/* Configuration de l'action command */
		jMenuItem.setActionCommand("showView."+name);
		/* Ajout du listener */
		jMenuItem.addActionListener(this);
		/* Ajout du JMenuItem au JMenu */
		jMenuShowView.add(jMenuItem);
		
	}
	
	/** Enlve toute les vues prsentes dans le menu lorsqu'il est ncessaire de les effacer
	 *  (changement de modle, ...)
	 */
	public void clearMenuShowView() {
		/* Retrait de tout les JMenuItem */
		jMenuShowView.removeAll();
	}

	/**
	 * Set the title of the parent JFrame 
	 * @param string : the tilte of the JFrame
	 */
	public void setTitle(String string) {
		
		/* Rcupration du container de plus haut niveau */
		Container cont = SwingUtilities.getAncestorOfClass(JFrame.class, this);
		/* Castage en JFrame */
    	JFrame jFrame = (JFrame) cont;
    	/* Configuration du nom de la fentre */
    	jFrame.setTitle("Moduleco - "+Moduleco.getCurrentModelName());
		
	}
	
}
