/*
* Role.java - Kernel: the kernel of MadKit
* Copyright (C) 1998-2008 Olivier Gutknecht, Fabien Michel, Jacques Ferber
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 2.1 of the License, or (at your option) any later version.

* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.

* You should have received a copy of the GNU Lesser General Public
* License along with this library; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
package madkit.kernel;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;

/**
 * Role class for MadKit 3.0
 * 
 * @author Fabien Michel
 * @version 3.0
 * @since MadKit 3.0
 */

@SuppressWarnings("serial")
final class Role extends HashSet<AgentAddress>
{
	private transient HashSet<Overlooker<? extends AbstractAgent>> overlookers;
	private transient ArrayList<AbstractAgent> referenceableAgents;
	private transient ArrayList<AbstractAgent> tmpReferenceableAgents;
	private transient boolean modified=false;
	
Role()
{
	referenceableAgents = new ArrayList<AbstractAgent>();
	tmpReferenceableAgents = new ArrayList<AbstractAgent>();
	overlookers = new HashSet<Overlooker<? extends AbstractAgent>>();
}

Role(final int size)
{
	super(size);
	referenceableAgents = new ArrayList<AbstractAgent>(size);
	tmpReferenceableAgents = new ArrayList<AbstractAgent>(size);
	overlookers = new HashSet<Overlooker<? extends AbstractAgent>>();
}

///////////////////////////////////////////////////////		ADD & REMOVE (also update overlookers)

/**
 * add the agent to the role
 * 
 * @param agent the agent
 * 
 * @return true, if the agent has been added.
 */
synchronized boolean addMember(final AgentAddress agent) {
	if(add(agent))
	{
		final AbstractAgent theReference = Kernel.getReference(agent);
		if(theReference instanceof ReferenceableAgent ) {
			referenceableAgents.add(theReference);
			updateOverlookers(theReference, true);
		}
		return true;
	}
	return false;
}

/**@return true if the role is no more useful: no member and not overlooked*/
synchronized boolean removeMember(final AgentAddress agent) {
	if(super.remove(agent)) {
		final AbstractAgent theReference = Kernel.getReference(agent);
		if(referenceableAgents.remove(theReference)){
			updateOverlookers(theReference,false);
		}
	}
	else
		return false;
	return isEmpty();
}

/**@return true if the role is no more useful: no member and not overlooked
 * @throws RoleNotHandled */
synchronized boolean removeMemberAndCheck(final AgentAddress agent) throws RoleNotHandled
{
	if(super.remove(agent)) {
		final AbstractAgent theReference = Kernel.getReference(agent);
		if(referenceableAgents.remove(theReference)){
			updateOverlookers(theReference,false);
		}
	}
	else
		throw new RoleNotHandled();
	return isEmpty();
}

@Override
public boolean isEmpty() {
	return super.isEmpty() && overlookers.isEmpty();
}

@Override
public boolean remove(Object agent) {
	if(super.remove(agent)) {
		final AbstractAgent theReference = Kernel.getReference(agent);
		if(referenceableAgents.remove(theReference)){
			updateOverlookers(theReference,false);
		}
		return true;
	}
	return false;
}

/**
 * @param theReference
 */
private void updateOverlookers(final AbstractAgent theReference,final boolean added) {
	modified=true;
	for (final Overlooker<? extends AbstractAgent> o : overlookers){
		o.update(theReference,added);// TODO choose solution on updateAgent
	}
}


/////////////////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////	OVERLOOKER PART

/**
 * Adds the overlooker.
 * 
 * @param o the o
 * 
 * @return true if the overlooker has been successfully added to the role
 */
synchronized boolean addOverlooker(final Overlooker<? extends AbstractAgent> o)
{
	if(overlookers.add(o))
	{
		o.setOverlookedRole( this );
		return true;
	}
	return false;
}

/**
 * Removes the overlooker from the role.
 * 
 * @param theOverlooker the overlooker to remove
 * 
 * @return true if, after the remove operation, the role is no more useful: no member and not overlooked.
 */
synchronized boolean removeOverlooker(final Overlooker<? extends AbstractAgent> theOverlooker)
{
	if(overlookers.remove(theOverlooker))
		return (isEmpty() && overlookers.isEmpty());
	return false;
}

synchronized List<AbstractAgent> getAgentsList()
{
	if(modified){
		modified=false;//TODO do a bench : new seems a little bit better
		//long startTime = System.nanoTime();
		//return Collections.unmodifiableList(referenceableAgents);
		tmpReferenceableAgents = new ArrayList<AbstractAgent>(referenceableAgents);
		//tmpReferenceableAgents = (ArrayList<AbstractAgent>)referenceableAgents.clone();
	   //long estimatedTime = System.nanoTime() - startTime;	   System.err.println(estimatedTime);

	}
	return tmpReferenceableAgents;
}

synchronized AbstractAgent getAgentNb(final int number) {
		return referenceableAgents.get(number);
}

///////////////////////////////////////////////:	UTILITY WHEN IMPORTING
void update()
{
	referenceableAgents = new ArrayList<AbstractAgent>();
	overlookers = new HashSet<Overlooker<? extends AbstractAgent>>(7);
}

@Override
public void clear() {
	referenceableAgents = null;//.clear();
	tmpReferenceableAgents = null;//.clear();
	for (Overlooker<? extends AbstractAgent> o : overlookers) {
		o.setOverlookedRole(null);
	}
	overlookers = null;//.clear();
	super.clear();
}
}
