/*
* Overlooker.java - Kernel: the kernel of MadKit
* Copyright (C) 1998-2008 Olivier Gutknecht, Fabien Michel, Jacques Ferber
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 2.1 of the License, or (at your option) any later version.

* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.

* You should have received a copy of the GNU Lesser General Public
* License along with this library; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
package madkit.kernel;

import java.util.Collections;
import java.util.List;
import java.util.ListIterator;

/**
 * This kernel internal tool defines the superclass class for probe and activator.
 * An overlooker is configured according to a community, a group and a role.
 * The agents variable is automatically updated and the update()
 * operation is invoked when changes occur on the given group and role.
 * 
 * @author Fabien Michel
 * @since MadKit 2.1
 * @version 3.0
 */

abstract class Overlooker <A extends AbstractAgent>
{ 
	private Role overlookedRole;
	final String community;
	final String group;
	final String role;
    
Overlooker(String communityName, String groupName, String roleName)
{
	community=communityName;
	group=groupName;
	role=roleName;
}
    
Overlooker(String group, String role)
{
	this(Kernel.DEFAULT_COMMUNITY, group, role);
}
    
synchronized final void setOverlookedRole(Role theRole)
{
	if(theRole != null){
		overlookedRole = theRole;
		initialize();
	}
}

abstract public void initialize();

void updateAgent(final AbstractAgent theAgent, final boolean added) {
	update(theAgent,added);
}
abstract public void update(final AbstractAgent theAgent,boolean added);

/** @return a ListIterator over the agents that handle the group/role couple
 * @deprecated as of MadKit 4.3 : simply type <code>getCurrentAgentsList().listIterator()</code>
@since MadKit 3.0*/
final public ListIterator<A> getAgentsIterator() 
{
	return getCurrentAgentsList().listIterator();
}

/** @return a ListIterator which has been previously shuffled
@since MadKit 3.0*/
final public List<A> getShuffledList()
{
	List<A> l = getCurrentAgentsList();
	Collections.shuffle(l);
	return l;
}

/** @return a list view (a snapshot at moment t) of the agents that handle the group/role couple (in proper sequence)
@since MadKit 3.0
*/
@SuppressWarnings("unchecked")
final synchronized public List<A> getCurrentAgentsList()
{
	return (List<A>)overlookedRole.getAgentsList();
}

/** @return the number of the agents that handle the group/role couple*/
final public int numberOfAgents()
{
	return overlookedRole.getAgentsList().size();
}

@SuppressWarnings("unchecked")
final synchronized public A getAgentNb(int nb)
{
	return (A) overlookedRole.getAgentNb(nb);
}

final public String getCommunity()  {	return community;   }
final public String getGroup()  {	return group;   }
final public String getRole()   {	return role;    }

@Override
public String toString()
{
	return this.getClass().getSimpleName()+" on <"+community+";"+group+";"+role+">, agents "+"are "+getCurrentAgentsList().size();
}


}
