/*
 * Kernel.java - Kernel: the kernel of MadKit
 * Copyright (C) 1998-2008 Olivier Gutknecht, Fabien Michel, Jacques Ferber
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package madkit.kernel;

import java.awt.Dimension;
import java.awt.Point;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;
import java.util.Vector;

import madkit.boot.Madkit;

//added by Saber so it will be possible to launch the netAgent while enabling mobility.
//import madkit.netcomm.NetAgent;



/** This class is the heart of the MadKit micro-kernel.  Most of these
  methods will only be useful to "system" agents developpers
  @author Ol Gutknecht
  @author Fabien Michel since MadKit 3.0
  	- org operations are now implemented in the Organization class
  	- there are also changes in messaging routines
  	- hooks are now called only when the considered operation is effectively done
  	- new features have been added:
  	  - migration of agents (for the moment for security reason, this feature has been
            more or less disabled).
  	  - communities
  @version 4.4
 */

final public class Kernel
{
	static public boolean debug = false;

	final static boolean onLineMode = true;


	final public static String VERSION = "4.4.0 - Exodus";
	final public static String BUGREPORT = "Please file bug reports on the madkit forum at http://www.madkit.net";

	final static String COMMUNITIES = "communities".intern();
	final public static String DEFAULT_COMMUNITY = "public".intern();

	final private static ThreadGroup systemAgentsThread=new ThreadGroup("System agents");
	final private static ThreadGroup regularAgentsThread=new ThreadGroup("Regular agents");
	final private static ThreadGroup schedulerAgentsThread=new ThreadGroup("Scheduler agents");

	public final static int SEND_MESSAGE            = 1;
	public final static int SEND_BROADCAST_MESSAGE  = 2;
	public final static int KILL_AGENT              = 3;
	public final static int CREATE_GROUP            = 4;
	public final static int LEAVE_GROUP             = 5;
	public final static int ADD_MEMBER_ROLE         = 6;
	public final static int REMOVE_MEMBER_ROLE      = 7;
	public final static int RESTORE_AGENT	    	= 8;
	public final static int LAUNCH_AGENT            = 0;
	public final static int CONNECTED_TO            = 9;
	public final static int DISCONNECTED_FROM       = 10;
	public final static int NEW_COMMUNITY	    	= 11;
	public final static int DELETE_COMMUNITY        = 12;
	public final static int MIGRATION = 13;

	final static int HOOKS_NUMBER = 14;

	public final static int GET_GROUPS    = 20;
	public final static int GET_AGENTS    = 21;
	public final static int DUMP_ORGANIZATION = 22;
	public final static int GET_AGENTINFO = 23;
	/*public final static int BE_COMMUNICATOR = 15;
	public final static int STOP_COMMUNICATOR = 16;*/
	public final static int CONNECTION = 24;
	public final static int DECONNECTION = 25;
	public final static int DUMP_COMMUNITIES = 27;
	//public final static int GET_AVAILABLE_DESTINATIONS = 26;

	static KernelAddress kernelAddress;

	private String kernelName;		//remove it ?? No it is now the name of the agency when mobility is activated!

	private Writer ostream = new OutputStreamWriter (System.err);

	private SiteAgent siteAgent;
	private KernelAgent kernelAgent;
	GraphicShell gui = null;

	private static Map<AgentAddress, AbstractAgent> localAgents = null;
	private Map<String, Organization> organizations;


	public static boolean agressiveHeapMode=true;
	public static int defaultAgentsAllocation=150000;
	static boolean interGroupMessage=true;

	static int agentsNb=0;

	public static int getAgentsNb()  { return agentsNb;  }  

	public static KernelAddress getAddress()  {    return kernelAddress;  }
	//private static void setAddress(KernelAddress k)  {    kernelAddress=k;  }

	/*	
		Added by Saber, modifs by JF
		net : the NetAgent launched when mobility is activated. 
		JF: it is now described by an interface.

	 */
	static Communicator net=null;
	private Keeper KeeperAgent=null;


	public void registerGUI(GraphicShell g)
	{
		gui=g;
		//siteAgent.initGUI();
		//disposeGUIOf(siteAgent);
		//redisplayGUIOf(siteAgent);
	}
	public String getName()    {	return kernelName;    }

	public Kernel(String theName, boolean ipnumeric)
	{
		kernelAddress = new KernelAddress(ipnumeric);
		kernelName=theName;
		initialization();
	}

	public Kernel(String theName, boolean ipnumeric, String ipaddress)
	{
		kernelAddress = new KernelAddress(ipnumeric,ipaddress);
		kernelName=theName;
		initialization();
	}

	public Kernel(String theName)
	{
		this(theName, false);
	}

	void initialization() 
	{
		if(! onLineMode)
			kernelAddress = new LocalKernelAddress();

		systemAgentsThread.setMaxPriority(Thread.MAX_PRIORITY);
		regularAgentsThread.setMaxPriority(Thread.NORM_PRIORITY);
		schedulerAgentsThread.setMaxPriority(Thread.NORM_PRIORITY-1);

		organizations = new Hashtable<String, Organization>();

		kernelAgent = new KernelAgent(); //a fake reference to pass
		siteAgent = new SiteAgent(organizations, kernelAgent);

		if(agressiveHeapMode)
			localAgents = new HashMap<AgentAddress, AbstractAgent>(defaultAgentsAllocation);
		else
			localAgents = new HashMap<AgentAddress, AbstractAgent>();

		try {
			launchAgent(siteAgent,"SITEAGENT"/*+kernelAddress*/,this,false);
		} catch (LaunchAgentException e) {
			display("!!!!!!!!!!!!!!!   system unstable !!!!!!!!!!!!!! ");
			e.printStackTrace();
		}

		System.err.println("\n\t-----------------------------------------------------");
		System.err.println("\n\t\t\tMadKit/Aalaadin \n\n\t\t  by MadKit Team (c) 1997-2008\n");
		System.err.println("\t\t     version: "+VERSION);
		System.err.println("\n\t-----------------------------------------------------\n");
		System.err.println(BUGREPORT+"\n");
		displayln("MadKit Agent microKernel "+getAddress() + " is up and running\n");
	}

	///////////////////////////////////// 	FUNDAMENTAL OPERATIONS : CREATE, REQUESTROLE, LEAVEROLE & LEAVEGROUP

	final synchronized void createGroup(AgentAddress creator, boolean distributed, String communityName, String groupName, String description, GroupIdentifier theIdentifier) throws OrganizationRequestException
	{
		Organization organization = getOrganizationFor(communityName);
		if(organization == null) // if the community does not exist, the request will always succeed. Could thus remove the check but this is not a costly operation
			organization = siteAgent.createCommunity(communityName);
		if(organization.createGroup(creator, distributed, groupName, description, theIdentifier)){
			kernelAgent.callHooks(CREATE_GROUP, creator, communityName, groupName, null);
			if(distributed)
				siteAgent.updateDistantOrgs(new SynchroMessage(creator,communityName,organization.get(groupName),groupName));
		}
		else{
			throw new CreateGroupException(creator," Group <"+communityName+","+groupName+"> : already exists (set agent debug on for a trace)");
		}
	}

	//	saber contribution
	final int joinPlace(Agent agent, String place, String pwd)
	{
		return(joinPlace(agent,"public",place,pwd));
	}

	final int joinPlace(Agent agent,String Community ,String place, String pwd)
	{
		return KeeperAgent.joinPlace(agent,Community,place,pwd);
	}

	final int createPlace(AgentAddress agent,String place, String informations)
	{
		if(!isGroup("public","Mobility")) // You must belong to a mobilityApplication before
		{
			System.err.println("This application is not yet able to be a mobile one ! You must create a mobile Group first.");
			return -1;
		}
		else if(isGroup("public",place))
		{
			System.err.println("Place : "+place+" already existing!");
			return -2;// Error_Place_AlreadyExisting
		}
		else
		{
			//if(AgencyKeeper.allowCreatePlace(AgentAddress)==true) sinon creation failed
			System.err.println("Creating the placeKeeper");
			PlaceKeeper pK = null; //new placeKeeper(this,place,KeeperAgent);
			try {
				// JF: instanciation of the PlaceKeeper through reflection
				//KeeperAgent = new AgencyKeeper(getAddress(), "agencyKeeperOf : "+KernelName);		
				Class<?> keeperClass = Utils.loadClass("madkit.mobility.PlaceKeeperAgent");
				Constructor<?> keeperCons = keeperClass.getConstructor(
						new Class[]{Kernel.class,String.class});//,Keeper.class});
				Agent ag = (Agent) keeperCons.newInstance(new Object[]{this,place});//,KeeperAgent});			
				pK = (PlaceKeeper) ag;
				launchAgent(ag,place+":placeKeeper",this,true);
			} catch (Exception e){
				System.err.println("Cannot create a placeKeeper: check if the Mobility plugin is available");
				return -1;
			}
			if(KeeperAgent.addPlace(pK,place))
			{
				return 1; // succeeded
			}
			else
			{
				killAgent((Agent) pK);
				return -1;
			}
		}
	}

	/*
final int createPlace(AbstractAgent agent,String place, String pwd)
{
			if(!isGroup("public","Mobility")) // You must belong to a mobilityApplication before
			{
					System.err.println("This application is not yet able to be a mobile one ! You must create a mobile Group first.");
					return -1;// Error_MobilytGroup_NotFound
			}
			else if(isGroup("public",place))
			{
					System.err.println("Place : "+place+" already existing!");
					return -2;// Error_Place_AlreadyExisting
			}
			else
			{
					placeKeeper kP = new placeKeeper(place,pwd);
					System.err.println("Place : "+place+" created!");
					return 1;//succeed
			}
}
	 */

	final boolean enableMobility(String KernelName, int port) // this is the first point making the application mobile
	{
		if(KeeperAgent==null) // Verify if there is already a mobility group or not
		{	try {
			// JF: instanciation of the KeeperAgent through reflection
			//KeeperAgent = new AgencyKeeper(getAddress(), "agencyKeeperOf : "+KernelName);		

			Class<?> keeperClass = Utils.loadClass("madkit.mobility.AgencyKeeper");
			Constructor<?> keeperCons = keeperClass.getConstructor(
					new Class[]{KernelAddress.class,String.class});
			Agent ag = (Agent) keeperCons.newInstance(new Object[]{getAddress(),"agencyKeeperOf : "+KernelName});			
			KeeperAgent = (Keeper) ag;
			launchAgent(ag,"agencyKeeperOf"+getAddress(),this,true);
		} catch (Exception e){
			System.err.println("Cannot create an AgencyKeeper: check if the Mobility plugin is available");
			return false;
		}

		kernelName=KernelName;
		kernelAddress.setKernelName(kernelName);
		kernelAddress.enableMobility();
		boolean status = launchCommunicator(port);

		if (status)
			System.err.println("Mobiliy activated : "+getAddress().getInformation());
		else
			System.err.println("Cannot activate mobility...");
		return status;
		}
		else
		{
			System.err.println("A mobility group is already existing!");
			return true;
		}
	}

	//	instanciation of a communicator through reflection... 
	boolean launchCommunicator(int port)
	{
		if(net==null)
		{
			try
			{
				Agent ag = null;
				Class<?> netClass = Utils.loadClass("madkit.netcomm.NetAgent");
				if (port == 0)
				{
					ag = (Agent) netClass.newInstance();
				} 
				else 
				{
					Constructor<?> construc = netClass.getConstructor(new Class[]{int.class});
					ag = (Agent) construc.newInstance(new Object[]{new Integer(port)});
				}
				net = (Communicator) ag;
				launchAgent(ag,"SiteAgent",this,true);
				return true;
			}
			catch(Exception e)
			{
				System.err.println("Cannot create a communicator agent"+e.toString());
			}
			return false;
		} else 
			return true;
	}

	final boolean supportMobility()
	{
		return kernelAddress.supportMobility();
	}

	final boolean connectAgencyToAgency(String host,int port)
	{
		if(supportMobility())
		{	

			if(net==null)
			{
				//				net = new NetAgent();
				//				launchAgent(net,"NetAgent",this,true);
				launchCommunicator(port);
			}	
			net.connectAgency(kernelName,host,port);
			return true;
		}
		System.err.println("Mobility is not supported yet!! ");
		return false;
	}			

	public KernelAddress getAgencyNamed(String name)
	{
		return KeeperAgent.getAgencyNamed(name);
	}


	//	Is
	public synchronized void launchMirror(Mirror mirror, AgentAddress address,String name, Object creator, boolean startGUI)
	{
		Point position = new Point(100,100);
		Dimension dim = new Dimension(120,120);
		Agent agent = (Agent) mirror;

		if (debug)
			displayln("Agent launch: "+ name +", created by " + creator.toString());
		if (agent.getAddress() == null)
			//		if (agent.getAgentInformation() == null)
		{
			agent.setMyAddress(new AgentAddress(name,Kernel.getAddress()));
			//agent.setAgentInformation(new AgentInformation(name, creator));
			agent.setCurrentKernel(this);
			agent.getAddress().update(address);
			//			System.out.println("///On a modifie l'adresse de notre mirroir elle est "+agent.getAddress()+" La mienne est "+address);
			agent.setCurrentKernel(this);
			localAgents.put(agent.getAddress(),agent);
			///*organization.*/requestRole(agent.getAddress(), "public", "local","member",null);	//temporaire
			if (startGUI && (gui != null))
				gui.setupGUI(agent,position,dim);
			Thread thread = null;
			//System.err.println("launch "+ Thread.currentThread() +"  "+Thread.currentThread().getPriority());
			if (agent instanceof Agent)
			{
				thread = new Thread(regularAgentsThread, agent, agent.getName() + "_thread");
				//thread.setPriority(5);//Thread.MIN_PRIORITY);
				//System.err.println("launch "+ thread +"  "+thread .getPriority());
			}

			if (thread != null)
				thread.start();
			else
			{
				Controller c = agent.getController();
				if (c != null)
					c.activate();
				else
					agent.activate();
			}
			agentsNb++;
			kernelAgent.callHooks(LAUNCH_AGENT, agent.getAddress());
		}
		else
			if (debug)
				displayln("ASSERT: agent already registred");
	}

	final synchronized void receiveAgent(AbstractAgent agent)
	{
		Thread thread = null;
		// if (agent instanceof Agent) modified by Saber  to 	if (agent instanceof mobileAgent)
		// The reason is quite simple we can't receive an agent if he isn't a mobile one
		// The examples like ball and super ping Pong will not run FROM NOW on :(
		//
		// Saber modification
		if (agent instanceof Mobile)
		{
			thread = new Thread(/*tg,*/(Agent) agent, agent.getName() + "_thread");
			// We will verify if we are receiving one of our mobileAgents we must give him his BirthAddress unless a new one

			//(AbstractAgent)localAgents.get(agent)

			if(((Mobile)agent).getMyAgency().equals(kernelAddress))
			{
				System.out.println("One of my Mobile Agents is coming back. Welcome!");
				Mirror mirror = (Mirror)localAgents.get(((Mobile)agent).getMyBirthAddress());
				killAgent((Agent) mirror);
				//				System.out.println("On a tué l'agent mirroir mais on garde sa trace.");
				((Mobile)agent).setMirrorAgent(mirror);
				agent.getAddress().update(((Mobile)agent).getMyBirthAddress());// A refaire de façon à ce que seul le kernel del'aggent puisse faire cette op�ration
			}
			else
			{
				agent.getAddress().update(kernelAddress);
			}
			agent.setCurrentKernel(this);
			localAgents.put(agent.getAddress(),agent);
			if (gui != null)
				gui.setupGUI(agent);
			if (thread != null)
				thread.start();
			else
				agent.activate();

		}
	}


	final void requestRole(AgentAddress requester, String communityName, String groupName, String roleName, Object memberCard) throws RequestRoleException
	{
		try {
			final Organization organization = getOrganizationFor(communityName);
			organization.requestRole(requester, groupName, roleName, memberCard);
			if(organization.isDistributed(groupName))
				siteAgent.updateDistantOrgs(new SynchroMessage(ADD_MEMBER_ROLE, requester, communityName, groupName, roleName, memberCard));
			kernelAgent.callHooks(ADD_MEMBER_ROLE, requester, communityName, groupName, roleName);
		} catch (NullPointerException e) {
			throw new RequestRoleException(requester,new CGRException(-1,communityName,null,null,e));
		} catch (CGRException e) {
			e.setCommunity(communityName);
			throw new RequestRoleException(requester,e);
		}
	}

	//	backup
	//	final int requestRole(AgentAddress requester, String communityName, String groupName, String roleName, Object memberCard)
	//	{
	//	final Organization organization = getOrganizationFor(communityName);
	//	if(organization != null)
	//	{
	//	int result = organization.requestRole(requester, groupName, roleName, memberCard);
	//	if(result == 1)
	//	{
	//	if(organization.isDistributed(groupName))
	//	siteAgent.updateDistantOrgs(ADD_MEMBER_ROLE, requester, communityName, groupName, roleName, memberCard);
	//	kernelAgent.callHooks(ADD_MEMBER_ROLE, requester, communityName, groupName, roleName);
	//	}
	//	return result;
	//	}
	//	else
	//	return -4;
	//	}

	final boolean leaveGroup(AgentAddress requester, String communityName, String groupName) throws LeaveGroupException
	{
		final Organization organization = getOrganizationFor(communityName);
		try {
			if(organization.leaveGroup(requester,groupName))
				siteAgent.removeCommunity(communityName);
		} catch (NullPointerException e) {
			throw new LeaveGroupException(requester, new CGRException(-1,communityName,null,null,e));
		} catch (CGRException e) {
			e.setCommunity(communityName);
			throw new LeaveGroupException(requester,e);
		}
		if(organization.isDistributed(groupName))
			siteAgent.updateDistantOrgs(new SynchroMessage(LEAVE_GROUP, requester,communityName, groupName, null, null));//creation inutile dans certains cas; à optimiser
		kernelAgent.callHooks(LEAVE_GROUP, requester, communityName, groupName, null);
		return true;


		//		if(organization != null && organization.isPlayingRole(requester,groupName,"member"))
		//		{
		//		if(organization.leaveGroup(requester,groupName))
		//		siteAgent.removeCommunity(communityName);
		//		}
		//		if(organization.isDistributed(groupName))
		//		siteAgent.updateDistantOrgs(LEAVE_GROUP, requester,communityName, groupName, null, null);
		//		kernelAgent.callHooks(LEAVE_GROUP, requester, communityName, groupName, null);
		//		return true;
	}

	final void leaveRole(AgentAddress requester, String communityName, String groupName, String roleName) throws LeaveRoleException
	{	
		final Organization organization = getOrganizationFor(communityName);
		try {
			final boolean removeCommunity = organization.leaveRole(requester,groupName,roleName);
			if(organization.isDistributed(groupName))
				siteAgent.updateDistantOrgs(new SynchroMessage(REMOVE_MEMBER_ROLE, requester,communityName, groupName, roleName, null));
			kernelAgent.callHooks(REMOVE_MEMBER_ROLE, requester, communityName, groupName, roleName);
			if (removeCommunity) {
				siteAgent.removeCommunity(communityName);
			}
		} catch (NullPointerException e) {
			throw new LeaveRoleException(requester, new CGRException(-1,communityName,null,null,e));
		} catch (CGRException e) {
			e.setCommunity(communityName);
			throw new LeaveRoleException(requester,e);
		}
		//return true;
		//TODO traiter l'exception ds le siteAgent 

		//		if(organization != null)// && organization.isPlayingRole(requester,groupName,roleName))
		//		{
		//		if(organization.isDistributed(groupName))
		//		siteAgent.updateDistantOrgs(REMOVE_MEMBER_ROLE, requester,communityName, groupName, roleName, null);
		//		kernelAgent.callHooks(REMOVE_MEMBER_ROLE, requester, communityName, groupName, roleName);
		//		return true;
		//		}
		//		else
		//		throw (new OrganizationRequestException(null,null,null,null));
		//		return false;
	}

	final boolean isBelongingToGroup(AgentAddress who, String communityName, String groupName)
	{
		try {
			return getOrganizationFor(communityName).isPlayingRole(who,groupName,Group.MEMBER_DEFAULT_ROLE);
		} catch (NullPointerException e) {
			return false;
		}
	}



	///////////////////////////////////////////////////////// OVERLOOKING

	/////////////////////////////////////////////////////////////////////////////////////////
	final boolean addOverlooker(AgentAddress requester, Overlooker<? extends AbstractAgent> o, Object accessCard)
	{
		try {
			return getOrganizationFor(o.community).addOverlooker(requester, o, accessCard);
		} catch (NullPointerException e) {
			return false;
		}
	}

	final void removeOverlooker(Overlooker<? extends AbstractAgent> o)
	{
		try {
			if(getOrganizationFor(o.community).removeOverlooker(o))
				siteAgent.removeCommunity(o.community);
		} catch (NullPointerException e) {
			System.err.println("error while removing overlooker "+e.getStackTrace());;
		} catch (Exception e) {
			System.err.println("error while removing overlooker "+e.getStackTrace());;
		}
	}
	///////////////////////////////////////////////////////////////




	//////////////////////////////////////////AGENT LAUNCHING & KILLING



	public final synchronized void launchAgent(AbstractAgent agent, String name, Object creator, boolean startGUI) throws LaunchAgentException{
		launchAgent(agent,name,creator,startGUI,new Point(-1,-1),new Dimension(-1,-1));
	}

	/**
	 * Launch an agent.
	 * 
	 * @param agent the agent
	 * @param name the name
	 * @param creator the creator
	 * @param startGUI the start gui
	 * @param position the position
	 * @param dim the dim
	 * @throws LaunchAgentException 
	 */
	final public synchronized void launchAgent(AbstractAgent agent, String name, Object creator, boolean startGUI, Point position, Dimension dim) throws LaunchAgentException
	{
		//		if (agent.getAgentInformation() == null)
		if (agent.getAddress() == null)
		{
			//agent.setAgentInformation(new AgentInformation(name, creator));
			agent.setMyAddress(new AgentAddress(name,Kernel.getAddress()));
			agent.setCurrentKernel(this);
			localAgents.put(agent.getAddress(),agent);
			///*organization.*/requestRole(agent.getAddress(), "public", "local","member",null);	//temporaire
			if (startGUI && (gui != null))
				gui.setupGUI(agent,position,dim);
			Thread thread = agent.getAgentThread();
			//System.err.println("launch "+ Thread.currentThread() +"  "+Thread.currentThread().getPriority());
			if (thread != null)
			{
				if(! (agent instanceof Scheduler)){
					if(! (agent instanceof SiteAgent || agent instanceof KernelAgent) ){
						thread = new Thread(regularAgentsThread,(Agent) agent, agent.getName() + "_thread");
						thread.setPriority(Thread.NORM_PRIORITY-1);
					}
					else{
						thread = new Thread(systemAgentsThread,(Agent) agent, agent.getName() + "_thread");
						thread.setDaemon(true);
						thread.setPriority(Thread.NORM_PRIORITY);//Thread.MIN_PRIORITY);
					}
				}
				else{
					thread = new Thread(schedulerAgentsThread,(Agent) agent, agent.getName() + "_thread");
					thread.setPriority(Thread.NORM_PRIORITY-2);				
				} 
			}

			if (thread != null)
				thread.start();
			else
			{
				Controller c = agent.getController();
				if (c != null)
					c.activate();
				else
					agent.activate();
			}
			agentsNb++;
			if (Madkit.debug){
				if(creator instanceof AbstractAgent)
					displayln("Agent launched : <"+ agent.getAddress() +">, created by agent <" + ((AbstractAgent)creator).getAddress()+">");
				else
					displayln("Agent launched : <"+ agent.getAddress() +">, created by <" + getName() + ((creator instanceof Kernel) ? ">" : ">, requested by "+ creator.getClass().getSimpleName()));
			}
			kernelAgent.callHooks(LAUNCH_AGENT, agent.getAddress());
		}
		else
			throw new LaunchAgentException(-30,agent.getAddress(), " unable to launch agent "+name+" : it is already registred",null);
	}

	/** Kill a given agent (from another agent). */
	final void killAgent(AbstractAgent target, AbstractAgent killer) throws KillAgentException
	{
		killAgent(target);
		//		try {
		//			if(target == killer || killer.getAddress().equals(target.getAgentInformation().getOwner()))
		//				killAgent(target);
		//			else
		//				throw new KillAgentException(-14,killer.getAddress()," failed to kill ["+target+"] : not the owner, I can only kill the agents I have launched!",null);
		//		} catch (NullPointerException e) {
		//			throw new KillAgentException(-15,killer.getAddress()," failed to kill ["+target+"] : this agent is null or already killed",e);
		//		}
		//		// Basically a wrapper verifying rights for the caller to terminate its mate */
		//		if (target != null && target.getAgentInformation() != null && (killer.getAddress().equals(target.getAgentInformation().getOwner()) || target == killer))
		//		killAgent(target);
	}

	/**
	 * Kill a given agent (also manage groups update).
	 * 
	 * @param theAgent the agent
	 */
	@SuppressWarnings("deprecation")
	final synchronized public void killAgent(AbstractAgent theAgent)
	{
		if(theAgent.getCurrentKernel() == null)
			return;
		final Thread t = theAgent.getAgentThread();
		if (t != null && t != Thread.currentThread()){
			t.stop();
		}
		//		if (theAgent instanceof Agent){
		//			Thread t = ((Agent)theAgent).getAgentThread();
		//			if(t != Thread.currentThread())
		//				t.stop();
		//		}
		Controller c = theAgent.getController();
		if (c != null)
			c.end();
		else
			theAgent.end();
		removeAgentFromOrganizations(theAgent.getAddress());
		localAgents.remove(theAgent.getAddress());
		if (gui != null)
			gui.disposeGUI(theAgent);
		theAgent.setCurrentKernel(null);
		kernelAgent.callHooks(KILL_AGENT, theAgent.getAddress());
		agentsNb--;
		//theAgent.setAgentInformation(null);
		theAgent.setMyAddress(null);
		theAgent = null;
		//debugString();
	}
	//	final synchronized public void killMirror(Mirror mirror)
	//	{
	//	if(mirror.getCurrentKernel() == null)
	//	return;
	//	if (Thread.currentThread() != mirror.getAgentThread())
	//	mirror.getAgentThread().stop();
	//	Controller c = mirror.getController();
	//	if (c != null)
	//	c.end();
	//	else
	//	mirror.end();

	//	removeAgentFromOrganizations(mirror.getAddress());
	//	localAgents.remove(mirror.getAddress());
	//	if (gui != null)
	//	gui.disposeGUI(mirror);
	//	mirror.setCurrentKernel(null);
	//	kernelAgent.callHooks(KILL_AGENT, mirror.getAgentInformation());
	//	agentsNb--;
	//	}

	final synchronized void restoreAgent(AbstractAgent agent, String name, Object creator, boolean startGUI)
	{
		if (debug)
			displayln("Agent restoration: "+ name +", restored by " + creator.toString());
		//if (agent.getAgentInformation() != null)
		if (agent.getAddress() != null)
		{
			//agent.setAgentInformation(new AgentInformation(name, creator));
			agent.setCurrentKernel(this);
			localAgents.put(agent.getAddress(),agent);

			/*organization.*/
			//requestRole(agent.getAddress(),"public", "local","member",null);		//temporaire

			if (startGUI && (gui != null))
				gui.setupGUI(agent);
			Thread thread = null;
			if (agent instanceof Agent)
				thread = new Thread((Agent) agent, agent.getName() + "_thread");
			if (thread != null)
				thread.start();
			else
				agent.activate();
			kernelAgent.callHooks(RESTORE_AGENT, agent.getAddress()); 
		}
		else
			if (debug)
				displayln("ASSERT: restoration impossible: agent has not been previously launched");
	}

	final synchronized private void killAgents()	// BUGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGG
	{
		for(Iterator<AgentAddress> i=localAgents.keySet().iterator(); i.hasNext(); ) {
			final AgentAddress next = i.next();
			if(! next.equals(kernelAgent.getAddress())) //Do not kill the KernelAgent
				removeAgentFromOrganizations(next);
			i.remove();
		}
	}

	final synchronized void removeReferenceOf(AgentAddress agent)
	{
		if (gui != null)
			try
		{
				gui.disposeGUI(localAgents.get(agent));
		}
		catch(Exception e){if (Madkit.debug) {
			e.printStackTrace();
		}}
		localAgents.remove(agent);
	}



	///////////////////////////////////////////////////////////////////////////////////////////

	////////////////////////////////////////////MESSAGING

	/** Send a message. If the the siteAgent has a communicator it will delegate the message to it
   @param sender Agent requesting the send
   @param receiver Destination agent
   @param m the Message itself
   @see Message*/
	final void sendMessage(Message m)
	{
		//		try
		//		{
		if (m.getReceiver().isLocal()) { // TODO could be good to do a bench with and without the hook call lines
			if (m instanceof PrivateMessage){
				sendLocalMessage(m);
			}
			else if (m instanceof SecuredMessage){
				sendLocalMessage(m.clone());
				kernelAgent.callHooks(SEND_MESSAGE, m.clone());
			}
			else{
				sendLocalMessage(m);
				kernelAgent.callHooks(SEND_MESSAGE, m);						
			}
			//kernelAgent.callHooks(SEND_MESSAGE, (m instanceof SecuredMessage) ? m.clone() : m);
		}	
		else
			//				if(siteAgent.sendDistantMessage(m)&& !(m instanceof PrivateMessage))
			//					kernelAgent.callHooks(SEND_MESSAGE, (m instanceof SecuredMessage) ? m.clone() : m);
			siteAgent.sendDistantMessage(m);
		//		}
		//		catch(MessageException mexc) {
		//			if (Madkit.debug){
		//				siteAgent.debug("Unable to send message "+m+" : "+mexc);
		//				mexc.printStackTrace();
		//			}
		//			throw(mexc);
		//		}
	}

	/** Sends a local message
   @param m the message itself
   @see Message*/
	final void sendLocalMessage(Message m) throws MessageException
	{
		try {
			//			if(m instanceof SecuredMessage)
			//				localAgents.get(m.getReceiver()).receiveMessage(m.clone());
			//			else
			localAgents.get(m.getReceiver()).receiveMessage(m);
		} catch (NullPointerException e) {
			throw new MessageException(m.getSender()," Unable to send message : unknown agent -> "+m.getReceiver(),(Madkit.debug)? e : null);
		}
	}

	/** Sends a broadcast message. If a specialized system agent can handle distributed message, the kernel will delegate the message to it
   @param groupName Group
   @param roleName Role
   @param m the Message itself
   @see Message*/
	final void sendBroadcastMessage(String communityName, String groupName, String roleName, Message m) throws MessageException
	{
		final Organization organization = getOrganizationFor(communityName);
		if(organization != null)
		{
			Collection<AgentAddress> receivers = null;
			synchronized (organization) {
				try {
					receivers = organization.getRolePlayers(groupName, roleName);
				} catch (CGRException e) {
					throw new MessageException(m.getSender()," Unable to broadcast message : "+e.getMessage(),(Madkit.debug)? e : null);
				}
				if(receivers == null)
					throw new MessageException(m.getSender()," Unable to broadcast message : nobody as receiver",null);
				if (!(m instanceof PrivateMessage))
				{
					ArrayList<Object> h = new ArrayList<Object>();
					h.add(groupName);
					h.add(roleName);
					if(m instanceof SecuredMessage)
						h.add(m.clone());
					else
						h.add(m);
					kernelAgent.callHooks(SEND_BROADCAST_MESSAGE, h);
				}

				try{
					if(Kernel.onLineMode){
						for(AgentAddress address : receivers){
							if (address.isLocal()){
								m = m.clone();
								m.setReceiver(address);
								sendLocalMessage(m);
							}
							else{
								m.setReceiver(address);
								siteAgent.sendDistantMessage(m);
							}
						}
					}
					else{
						for(AgentAddress address : receivers){
							m = m.clone();
							m.setReceiver(address);
							sendLocalMessage(m);
						}
					}
				}
				catch(MessageException mexc) {
					if (Madkit.debug){
						siteAgent.debug("Unable to send message : "+m+" : "+mexc);
						mexc.printStackTrace();
					}
				}
			}
		}
	}

	//////////////////////////////////////////////////internal methods

	final static synchronized AbstractAgent getReference(Object agent)
	{
		return localAgents.get(agent);
	}

	final static synchronized AgentAddress[] getLocalAgents()
	{
		return localAgents.keySet().toArray(new AgentAddress[localAgents.size()]);
	}



	///////////////////////////////////////////////// ORGANIZATIONS MANAGEMENT
	final Organization getOrganizationFor(String communityName)
	{
		return organizations.get(communityName);
	}

	final boolean isGroup(String community, String groupName)
	{
		try {
			return getOrganizationFor(community).getNotEmptyGroup(groupName) != null;
		} catch (NullPointerException e) {
			return false;
		}
	}

	//	final boolean isRole(String community, String groupName, String roleName)
	//	{
	//		try {
	//			return getOrganizationFor(community).get(groupName).getRolePlayers(roleName).size() > 0;
	//		} catch (NullPointerException e) {
	//		} catch (CGRException e) {
	//		}
	//		return false;
	//	}

	final synchronized public String[] getCurrentGroupsOf(AgentAddress theAgent, String communityName)
	{
		try {
			final Collection<String> c = getOrganizationFor(communityName).getCurrentGroupsOf(theAgent); 
			return c.toArray(new String[c.size()]);
		} catch (NullPointerException e) {
			return new String[0];
		}
	}

	final synchronized public String[] getExistingGroups(String communityName)
	{
		try {
			return getOrganizationFor(communityName).getGroups();
		} catch (NullPointerException e) {
			return new String[0];
		}
	}

	final synchronized public AgentAddress[] getRolePlayers(final String communityName, final String groupName, final String roleName) throws CGRException
	{
		try {
			Collection<AgentAddress> rolePlayers = getOrganizationFor(communityName).getRolePlayers(groupName, roleName); 
			return rolePlayers.toArray(new AgentAddress[rolePlayers.size()]);
		} catch (NullPointerException e) {
			throw new CGRException(-1,communityName,null,null,e);
		} catch (CGRException e) {
			e.setCommunity(communityName);
			throw(e);
		}
	}

	/*final synchronized AgentAddress getRolePlayer(final String communityName, final String groupName, final String roleName) throws CGRException
{
	try {
		return getOrganizationFor(communityName).getRolePlayer(groupName, roleName);
	} catch (NullPointerException e) {
		throw new CGRException(communityName,null,null,e);
	} catch (CGRException e) {
		e.setCommunity(communityName);
		throw(e);
	}
}
	 */

	final synchronized AgentAddress getAnotherRolePlayer(final AbstractAgent requester, final String communityName, final String groupName, final String roleName) throws CGRException
	{
		try {
			final java.util.List <AgentAddress>  rolePlayers = new java.util.ArrayList<AgentAddress>(getOrganizationFor(communityName).getRolePlayers(groupName, roleName));
			rolePlayers.remove(requester.getAddress());
			int size = rolePlayers.size();
			if(size == 0)
				return null;
			return rolePlayers.get((int) (Math.random()*size));
		} 
		catch (NullPointerException e) {
			throw new CGRException(-1,communityName,null,null,e);
		} catch (CGRException e) {
			e.setCommunity(communityName);
			throw(e);
		}
	}

	final synchronized String[] getGroupRolesOf(final AgentAddress agent, final String communityName, final String groupName)
	{
		try {
			return getOrganizationFor(communityName).getGroupRolesOf(agent, groupName);
		} catch (NullPointerException e) {
			return new String[0];
		}
	}

	final synchronized public String[] getExistingRoles(final String communityName, final String groupName)
	{
		try {
			return getOrganizationFor(communityName).getRolesIn(groupName);
		} catch (NullPointerException e) {
			return new String[0];
		}
	}

	final synchronized boolean isCommunity(String communityName)
	{
		final Organization organization = getOrganizationFor(COMMUNITIES);
		try {
			return  (organization.isPlayingRole(siteAgent.getAddress(),communityName,Group.MEMBER_DEFAULT_ROLE) || organization.getRolePlayer(communityName,"site") != null);
		} catch (CGRException e) {
			return false;
		}
	}

	final synchronized String[] getCommunities()
	{
		return siteAgent.getCommunities();
	}

	final synchronized boolean connectedWithCommunity(final String communityName)
	{
		return getOrganizationFor(COMMUNITIES).isPlayingRole(siteAgent.getAddress(), communityName, "site");
	}

	final synchronized void removeAgentFromOrganizations(final AgentAddress theAgent)	// must be optimized
	{
		final ArrayList<String> emptyOrgs = new ArrayList<String>();
		for (Map.Entry<String, Organization> entry : organizations.entrySet()) {
			Organization org = entry.getValue();
			if(! org.agentIn(theAgent))
				continue;
			String orgName = entry.getKey();
			for(String groupName : org.removeAgentFromAllGroups(theAgent)){
				if(org.isDistributed(groupName))
					siteAgent.updateDistantOrgs(new SynchroMessage(LEAVE_GROUP, theAgent,orgName, groupName, null, null));//creation inutile dans certains cas; à optimiser
				kernelAgent.callHooks(LEAVE_GROUP, theAgent, orgName, groupName, null);
			}
			if(org.isEmpty())
				emptyOrgs.add(orgName);
		}
		for (String string : emptyOrgs) {
			siteAgent.removeCommunity(string);
		}


		//		for (Map.Entry<String, Organization> entry : new HashMap<String, Organization>(organizations).entrySet()) {
		//			entry.getValue().removeAgentFromGroups(this,theAgent,entry.getKey());
		//		}
		//	




		//		final Map<String,Collection<String>> groupNames = new HashMap<String,Collection<String>>();
		//
		//		for (Map.Entry<String, Organization> entry : organizations.entrySet()) {
		//			groupNames.put(entry.getKey(),entry.getValue().getCurrentGroupsOf(theAgent));
		//		}
		//
		//		for (Map.Entry<String, Collection<String>> entry : groupNames.entrySet()) {
		//			for (String group : entry.getValue()) {
		//				try {
		//					leaveGroup(theAgent,entry.getKey(),group);
		//				} catch (LeaveGroupException e) {
		//					e.printStackTrace();
		//				}
		//			}
		//		}
	}

	synchronized Map<String,Map> getDumpCommunities(){
		final Map<String,Map> res = new HashMap<String,Map>();
		for(Map.Entry<String, Organization> e : organizations.entrySet()){
			res.put(e.getKey(), e.getValue().getLocalOrganization());
		}
		return res;
	}

	/////////////////////////////////////////////////////////////////////////////////////



	///////////////////////////////////////////////   Others
	/** Request a kernel stop. All agents are (hopefully) cleanly killed */
	public void stopKernel()
	{
		displayln("Disconnecting MadKit Kernel: " + getName());
		displayln("- killing local agents " + getName());
		killAgents();
		displayln("MadKit Kernel closed.");
		System.exit(0);
	}

	/** A generic display method adapting its output to the kernel environment (console, GUI, applet...)
	@param s string to be displayed, add a newline at the end of the string */
	public void displayln(String s)
	{
		display(s+'\n');
	}

	/** A generic display method adapting its output to the kernel environment (console, GUI, applet...)
	@param s string to be displayed */
	public void display(String s)
	{
		System.err.print("<" + getName() + "> : " + s);
	}

	/** Reassigns the "standard" agent text output stream (used by println method).  */
	public void setOutputStream(Writer o)  {    ostream = o;  }

	final void disposeGUIOf(AbstractAgent theAgent)
	{
		if (gui != null)
			gui.disposeGUIImmediatly(theAgent);
	}

	final void redisplayGUIOf(AbstractAgent theAgent)
	{
		if (gui != null)
			gui.setupGUI(theAgent);
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////:

	//////////////////////////////////////////////DEPRECATED METHODS

	/**@deprecated As of MadKit 3.0. replaced by {@link #getCurrentGroupsOf(AgentAddress, String)}*/

	//	final check before deletion 14 10 2007
	/*public Vector getCurrentGroups(AgentAddress theAgent)
{
	Vector v = new Vector();
	String[] groups = getCurrentGroupsOf(theAgent, Kernel.DEFAULT_COMMUNITY);
	for(int i = 0;i<groups.length;i++)
		v.addElement(groups[i]);
	return v;
}*/

	/**@deprecated As of MadKit 3.0. replaced by {@link #getMembersWithin(String, String)}*/
	/*synchronized public Enumeration getGroupMembers(String theGroup)
{
	Collection c = new HashSet();
	AgentAddress[] addresses = getMembersWithin(theGroup, Kernel.DEFAULT_COMMUNITY);
	for(int i=0;i< addresses.length;i++)
		c.add(addresses[i]);
	return Collections.enumeration(c);
}
	 */
	/**@deprecated As of MadKit 3.0. replaced by {@link #getGroupRolesOf(AgentAddress, String, String)}please use getGroupRolesOf instead*/
	@Deprecated
	synchronized public Vector getMemberRoles(String theGroup, AgentAddress theAgent)
	{
		Vector v = new Vector();
		String[] roles = getGroupRolesOf(theAgent, Kernel.DEFAULT_COMMUNITY, theGroup);
		for(int i = 0;i<roles.length;i++)
			v.addElement(roles[i]);
		return v;
	}

	void synchronizeKernel(Map<String,Organization> orgs, boolean priority)
	{
		for(Map.Entry<String,Organization> e :orgs.entrySet())
			if(siteAgent.connectedWith(e.getKey()))
			{
				if (! organizations.containsKey(e.getKey()))
					organizations.put(e.getKey(),new Organization());
				( organizations.get(e.getKey()) ).importOrg(e.getValue(),priority);
			}
		siteAgent.refreshCommunities();
	}

	public static void debugString()
	{
		System.err.println("--------------------------------------kernel status");
		ThreadGroup tg = Thread.currentThread().getThreadGroup();
		Thread[] temp=new Thread[tg.activeCount()];
		tg.enumerate(temp);
		for(int i = 0; i< temp.length; i++)
		{
			if (temp[i] != null)
				System.err.println(""+i+":  "+temp[i]+" is demon "+temp[i].isDaemon());
		}
		System.err.println("used memory: "+(Runtime.getRuntime().totalMemory()-Runtime.getRuntime().freeMemory()));
		System.err.println("locals agents are "+localAgents.size());
	}

	public void destroyGroup(String communityName, String groupName) {
		final Organization org = organizations.get(communityName); 
		org.remove(groupName).clear();
		if(org.isEmpty())
			siteAgent.removeCommunity(communityName);	
	}

}

