/*
 * AbstractAgent.java - Kernel: the kernel of MadKit
 * Copyright (C) 1998-2008 Olivier Gutknecht, Fabien Michel, Jacques Ferber
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package madkit.kernel;

import java.awt.AWTEvent;
import java.awt.Dimension;
import java.awt.Point;
import java.io.IOException;
import java.io.Serializable;
import java.io.Writer;

import madkit.boot.Madkit;

/**
 * The main MadKit AbstractAgent class. It provides support for agent's
 * <ul>
 * <li> Lifecycle
 * <li> Group and roles management
 * <li> Messaging
 * <li> Graphical interface
 * <li> Agent informations
 * </ul>
 *<p>
 * The agent's behavior is <i>intentionally not defined</i>. It is up to the
 * agent developer to choose an agent model or to develop his specific
 * agent library on top of the facilities provided by MadKit. However,
 * all agent share the same organizational view, and the basic
 * messaging code, so integration of agent coming from different
 * developer and having heterogeneous models is quite easy.
 *<p>
 * Agent-related methods (almost everything here) can be invoked only
 * after registration in the kernel (i.e. after the <code>activate</code> method has been invoked
 * on this agent). That means that you should not use any of the agent methods in constructor
 *
 * @author Olivier Gutknecht
 * @author Fabien Michel
 * @version 3.6	20/05/2008
 */

@SuppressWarnings("serial")
public class AbstractAgent extends Object implements Serializable
{
	transient private Object currentBean;
	transient private Writer ostream;

	private boolean debugFlag = false;
	//private AgentInformation agentInformation = null;
	private AgentAddress myAddress = null;

	transient Kernel currentKernel = null;

	final java.util.Vector<Message> messageBox;

	private Controller controller=null;
	//private boolean bean_mode;
	final private int _hashCode;
	static private int agentCounter=0;

	/** Default no-args constructor, which just set-up internal structures */
	public AbstractAgent()
	{
		_hashCode=++agentCounter;
		messageBox = new java.util.Vector<Message>();
	}

	/**
	 * Returns the current controller of the agent if there is one.
	 * 
	 * @return the controller
	 */
	public Controller getController(){return controller;}
	/** Assign a controller which will be in charge of the control of the agent's behavior*/
	public void setController(final Controller c){controller = c;}



	///////////////////////////////////////////////////////////////////////		BASIC COMMANDS
	/**
	 * This method is initially called when the micro-kernel registers the agent.
	 * Usually a good place to set up some initial groups and roles.
	 * <p>
	 * Here is a typical example (taken from the <code>PingPong</code> agent of MadKit (plugin demos)) :
	 * <p><blockquote><pre>
public void activate()
{
	try {
		createGroup(true,"ping-pong",null,null);
		println ("No ping-pong group : I created one");
		creator=true;
	} catch (OrganizationRequestException e) { // the group already exists
		creator=false;
		println ("A ping-pong group already exists : I will join");
	}
	requestRole("ping-pong","player",null);
}
	 * </pre></blockquote>
	 *
	 */
	public void activate() {}

	/** This method is called by the agent micro-kernel at the end of the agent lifecycle.
If the the agent is killed from the outside, it is the last opportunity
for the agent to cleanly shutdown its operations. Although it is "nicer" to make the agent leave its groups, the handled groups and roles are automatically leaved after this last step*/
	public void end() {}



	/////////////////////////////////////////////////////////// GROUP & ROLE METHODS	AGR
	/**
	 * Creates a new Group.
	 * If operation succeed, the agent will automatically handle two roles: <i>member</i> and <i>group manager</i>.
	 * 
	 * @param distributed if <code>true</code> the new group will be distributed when multiple MadKit kernels are connected.
	 * @param communityName the community within the group will be created. If this community does not exist it will be created.
	 * @param groupName the name of the new group
	 * @param description can be null (thus the description will be the name of the group)
	 * @param theIdentifier an object that implements the <code>GroupIdentifer Interface</code> which could be used to check if an agent can be admitted in the group. When this object is null, there is no group access control
	 * @return <ul>
	 * <li><code>  1 </code>: operation success;</li>
	 * <li><code> -8 </code>: operation failed ; the group already exists </li>
	 * </ul>
	 * 
	 * @see GroupIdentifier
	 * @since MadKit 3.0
	 */
	public int createGroup(final boolean distributed, final String communityName, final String groupName, String description, final GroupIdentifier theIdentifier){
		if (description == null)
			description = groupName;
		try {
			currentKernel.createGroup(getAddress(), distributed, communityName, groupName, description, theIdentifier);
		} catch (OrganizationRequestException e) {
			return handleException(e);
		}
		return 1;
	}

	/**
	 * Creates a new Group within the default community <i>public</i>. I.e. it will call createGroup(distributed, Kernel.DEFAULT_COMMUNITY, groupName, description, theIdentifier) 
	 * 
	 * @param distributed if <code>true</code> the new group will be distributed when multiple MadKit kernels are connected.
	 * @param groupName the name of the new group
	 * @param description can be null (thus the description will be the name of the group)
	 * @param theIdentifier an object that implements the <code>GroupIdentifer Interface</code> which could be used to check if an agent can be admitted in the group. When this object is null, there is no group access control
	 * @return <ul>
	 * <li><code>  1 </code>: operation success;</li>
	 * <li><code> -8 </code>: operation failed ; the group already exists </li>
	 * </ul>
	 * @see GroupIdentifier
	 * @since MadKit 3.0
	 */
	public int createGroup(final boolean distributed, final String groupName, final String description, final GroupIdentifier theIdentifier)
	{
		return createGroup(distributed, Kernel.DEFAULT_COMMUNITY, groupName, description, theIdentifier);
	}

	/**
	 * Requests a role within a group of a particular community.
	 * 
	 * @param communityName the group's community.
	 * @param groupName the desired group.
	 * @param roleName the desired role.
	 * @param memberCard the passKey to enter a group. If needed, it is generally delivered by the group's <i>group manager</i> to nice agents :)
	 * It can be <code> null </code> when the desired group has no security (i.e. was created using <code> null </code> for <i> theIdentifier </i> parameter).
	 * @return <ul>
	 * <li><code>  1 </code>: operation success;</li>
	 * <li><code> -1 </code>: the community does not exist.</li>
	 * <li><code> -2 </code>: the group does not exist;</li>
	 * <li><code> -4 </code>: the role is already handled by this agent;</li>
	 * <li><code> -5 </code>: access denied by the manager of that group;</li>
	 * </ul>
	 * @see GroupIdentifier
	 * 
	 * @since MadKit 3.0
	 */
	public int requestRole(final String communityName, final String groupName, final String roleName, final Object memberCard)	{
		try{
			currentKernel.requestRole(getAddress(), communityName, groupName, roleName, memberCard);
			return 1;
		}
		catch(RequestRoleException e){
			return handleException(e);
		}
	}
	/**
	 * Requests a role within a group of the default community.
	 * 
	 * @param groupName the group name
	 * @param roleName the role name
	 * @param memberCard the passKey to enter a group. If needed, it is generally delivered by the group's <i>group manager</i> to nice agents :)
	 * It can be <code> null </code> when the desired group has no security (i.e. was created using <code> null </code> for <i> theIdentifier </i> parameter).
	 * @return <ul>
	 * <li><code>  1 </code>: operation success;</li>
	 * <li><code> -2 </code>: the group does not exist;</li>
	 * <li><code> -4 </code>: the role is already handled by this agent;</li>
	 * <li><code> -5 </code>: access denied by the manager of that group;</li>
	 * </ul>
	 * @since MadKit 3.0
	 * @see GroupIdentifier
	 */
	public int requestRole(final String groupName, final String roleName, final Object memberCard) {
		return requestRole(Kernel.DEFAULT_COMMUNITY, groupName, roleName, memberCard);
	}

	/**
	 * Abandons an handled role within a group of a particular community.
	 * 
	 * @param communityName the community name
	 * @param groupName the group name
	 * @param roleName the role name
	 * @return <ul><li><code>  1 </code>: operation success;</li>
	 * <li> <code> -1 </code>: the community does not exist.</li>
	 * <li><code> -2 </code>: the group does not exist;</li>
	 * <li><code> -6 </code>: the role is not handled;</li>	 
	 * </ul>
	 * @since MadKit 3.0
	 */
	//* * @throws LeaveRoleException if<ul><li>the role is not handled</li><li>the community or the group does not exist</li></ul>
	public int leaveRole(final String communityName, final String groupName, final String roleName) {
		try {
			currentKernel.leaveRole(getAddress(), communityName, groupName, roleName);
			return 1;
		} catch (LeaveRoleException e) {
			return handleException(e);
		}
	}

	/**
	 * Abandons an handled role within a group of the default community.
	 * 
	 * @param groupName the group name
	 * @param roleName the role name
	 * 
	 * @return <ul><li><code>  1 </code>: operation success;</li>
	 * <li><code> -2 </code>: the group does not exist;</li>
	 * <li><code> -6 </code>: the role is not handled;</li>	 
	 * </ul>
	 * 
	 * @since MadKit 3.0
	 */

	public int leaveRole(final String groupName, final String roleName) {
		return leaveRole(Kernel.DEFAULT_COMMUNITY, groupName, roleName);
	}

	/**
	 * Makes the agent leave a group of a particular community.
	 * 
	 * @param communityName the community name
	 * @param groupName the group name
	 * 
	 * @return <ul><li><code>  1 </code>: operation success;</li>
	 * <li> <code> -1 </code>: the community does not exist;</li>
	 * <li><code> -2 </code>: the group does not exist;</li>
	 * <li><code> -7 </code>: the agent is not a member of that group</li>	 
	 * </ul>
	 * 
	 * @since MadKit 3.0
	 */
	public int leaveGroup(final String communityName, final String groupName)	{
		try {
			currentKernel.leaveGroup(getAddress(), communityName, groupName);
			return 1;
		} catch (LeaveGroupException e) {
			return handleException(e);
		}
	}

	/**
	 * Makes the agent leave a group of the default community.
	 * 
	 * @param groupName the group name
	 * 
	 * @return <ul><li><code>  1 </code>: operation success;</li>
	 * <li><code> -2 </code>: the group does not exist;</li>
	 * <li><code> -7 </code>: the agent is not a member of that group</li>	 
	 * </ul>
	 * 
	 */
	public int leaveGroup(final String groupName) {
		return leaveGroup(Kernel.DEFAULT_COMMUNITY, groupName);
	}



	///////////////////////////////////////////////////////////// DEPRECATED METHODS

	/** @deprecated As of MadKit 3.0. replaced by {@link #createGroup(boolean,String,String,String,GroupIdentifier)}.
	<p>
	This call is now equivalent to <code> createGroup(true, groupName, null, null)</code>
	if the group does not exist or <code> requestRole(groupName, "member", null)</code> otherwise.
	 */
	public void joinGroup(final String groupName)
	{
		if(currentKernel.isGroup(Kernel.DEFAULT_COMMUNITY, groupName))
			requestRole(Kernel.DEFAULT_COMMUNITY, groupName,Group.MEMBER_DEFAULT_ROLE,null);
		else
			createGroup(true, Kernel.DEFAULT_COMMUNITY, groupName, null, null);
	}
	/** @throws RequestRoleException 
	 * @deprecated As of MadKit 3.0. replaced by {@link #requestRole(String,String,Object)}.
	<p>
	This call is now equivalent to <code> requestRole(groupName, roleName, null)</code>.
	 */
	public void requestRole(final String groupName, final String roleName) throws RequestRoleException{currentKernel.requestRole(getAddress(),Kernel.DEFAULT_COMMUNITY, groupName, roleName, null);}

	/** @throws OrganizationRequestException 
	 * @deprecated As of MadKit 3.0. replaced by {@link #createGroup(boolean,String,String,String,GroupIdentifier)}.
	<p>
	This call is now equivalent to <code>createGroup(true, groupName, null, null)</code>.
	 */
	public void foundGroup(final String groupName) throws OrganizationRequestException{currentKernel.createGroup(getAddress(), true, Kernel.DEFAULT_COMMUNITY, groupName, groupName, null);}



	//////////////////////////////////////////////////////////ORGANIZATION INFORMATIONS

	/**
	 * Gets the addresses of all the agents (including this agent if present) that handle this role within this group in this community.
	 * 
	 * @param communityName the community name
	 * @param groupName the group name
	 * @param roleName the role name
	 * 
	 * @return the agents which handles this role (including this agent if present). An array of size 0 if no agent match
	 * 
	 */
	//	* @throws OrganizationRequestException if the CGR triple is not found
	public AgentAddress[] getAgentsWithRole(final String communityName, final String groupName, final String roleName) //throws OrganizationRequestException
	{
		try {
			return currentKernel.getRolePlayers(communityName, groupName, roleName);
		} catch (CGRException e) {
			return new AgentAddress[0];//throw new OrganizationRequestException(getAddress(),e);
		}
	}

	/**
	 * Gets the addresses of all the agents (including this agent if present) that handle this role within this group.
	 * 
	 * @param groupName the group name
	 * @param roleName the role name
	 * 
	 * @return the agents playing this role, or a 0 size array if there is no agent playing this role
	 * 
	 */
	public AgentAddress[] getAgentsWithRole(final String groupName, final String roleName) 
	{
		return getAgentsWithRole(Kernel.DEFAULT_COMMUNITY, groupName, roleName);
	}

	/**
	 * Gets an agent that handles a given role within a group of a particular community.
	 * The agent is <b><i>chosen randomly</i> among the other agents</b>
	 * 
	 * @param groupName group name
	 * @param roleName role name
	 * @param communityName community name
	 * 
	 * @return an agent address if available, <code>null</code> otherwise.
	 * 
	 * @since MadKit 3.0
	 */
	public AgentAddress getAgentWithRole(final String communityName, final String groupName, final String roleName) 
	{
		try {
			return currentKernel.getAnotherRolePlayer(this,communityName, groupName, roleName);
		} catch (CGRException e) {
			return null;
			//throw new OrganizationRequestException(getAddress(),e);
		}
	}

	/**
	 * Gets an agent that handles a given role within a group of the default MadKit community.
	 * The agent is <b><i>chosen randomly</i> among the other agents</b></b>
	 * 
	 * @param groupName the group name
	 * @param roleName the role name
	 * 
	 * @return an agent address if available, <code>null</code> otherwise.
	 */
	public AgentAddress getAgentWithRole(final String groupName, final String roleName)
	{
		return getAgentWithRole(Kernel.DEFAULT_COMMUNITY, groupName, roleName);
	}

	/**
	 * Gets the name of the groups the agent joined in this community.
	 * 
	 * @param communityName  a string holding a community name
	 * 
	 * @return an array containing the names of the groups the agent is in. It could be 0 size if there is no group
	 * 
	 * @since MadKit 3.0
	 */
	public String[] getMyGroups(final String communityName)
	{
		return currentKernel.getCurrentGroupsOf(getAddress(), communityName);
	}

	/**
	 * Gets the name of the groups the agent joined in the default community.
	 * 
	 * @return an array containing the names of the groups the agent is in. It could be 0 size if there is no group
	 * 
	 * @since MadKit 3.0
	 */
	public String[] getMyGroups()
	{
		return currentKernel.getCurrentGroupsOf(getAddress(), Kernel.DEFAULT_COMMUNITY);
	}

	/**
	 * returns the names of the groups that exist in this community.
	 * 
	 * @param communityName the community name
	 * 
	 * @return an array containing the names of the groups which exist in this community. It could be 0 size if there is no group, i.e. the community does not exist.
	 */
	public String[] getExistingGroups(final String communityName)
	{
		return currentKernel.getExistingGroups(communityName);
	}

	/**
	 * returns an array containing the names of the groups which exist in the default community. 
	 * 
	 * @return the existing groups 
	 */
	public String[] getExistingGroups()
	{
		return currentKernel.getExistingGroups(Kernel.DEFAULT_COMMUNITY);
	}

	/**
	 * Tells if the agent is a member of a specific group in a given community
	 * @return true if the agent is a member of that group
	 */
	public boolean isMemberOf(final String communityName, final String groupName)
	{
		return currentKernel.isBelongingToGroup(getAddress(), communityName, groupName);
	}

	/**
	 * Tells if the agent is a member of a specific group in the default community.
	 * 
	 * @param groupName the group name
	 * 
	 * @return true if the agent is a member of the group
	 */
	public boolean isMemberOf(final String groupName)
	{
		return currentKernel.isBelongingToGroup(getAddress(), Kernel.DEFAULT_COMMUNITY, groupName);
	}


	/**
	 * Tells if an agent is a member of a specific group in a given community.
	 * 
	 * @param address the agent address
	 * @param communityName the community name
	 * @param groupName the group name
	 * 
	 * @return true if the agent is member of a group
	 */
	public boolean isBelongingToGroup(final AgentAddress address, final String communityName, final String groupName)
	{
		return currentKernel.isBelongingToGroup(address, communityName, groupName);
	}

	/**
	 * Tell if an agent is a member of a specific group in the default community
	 * @return true if the agent is member of a group
	 */
	public boolean isBelongingToGroup(final AgentAddress address, final String groupName)
	{
		return currentKernel.isBelongingToGroup(address, Kernel.DEFAULT_COMMUNITY, groupName);
	}


	/**
	 * Gets roles currently handled within a group of this community.
	 * 
	 * @param communityName the community name
	 * @param groupName the group name
	 * 
	 * @return the my roles
	 */
	public String[] getMyRoles(final String communityName, final String groupName)
	{
		return currentKernel.getGroupRolesOf(getAddress(), communityName, groupName);
	}

	/**
	 * Gets roles currently handled within a group of the default community.
	 * 
	 * @param groupName the group name
	 * 
	 * @return the my roles
	 */
	public String[] getMyRoles(final String groupName)
	{
		return currentKernel.getGroupRolesOf(getAddress(), Kernel.DEFAULT_COMMUNITY, groupName);
	}

	/**
	 * returns the names of the roles that exist in this (community,group) couple.
	 * 
	 * @param communityName the community name
	 * @param groupName the group name
	 * 
	 * @return the existing roles
	 */
	public String[] getExistingRoles(final String communityName, final String groupName)
	{
		return currentKernel.getExistingRoles(communityName, groupName);
	}

	/**
	 * returns the names of the roles that exist within this group in the default community.
	 * 
	 * @param groupName the group name
	 * 
	 * @return the existing roles
	 */
	public String[] getExistingRoles(final String groupName)
	{
		return currentKernel.getExistingRoles(Kernel.DEFAULT_COMMUNITY, groupName);
	}

	/** @deprecated As of MadKit 3.0. replaced by {@link #getExistingRoles(String,String)}*/
	public String[] getRoles(final String groupName)
	{
		return getExistingRoles(Kernel.DEFAULT_COMMUNITY, groupName);
	}

	/** @return <code> true </code>if the role exists (i.e. there is at least one agent having this role); <code> false </code>otherwise.*/
	public boolean isRole(final String communityName, final String groupName, final String roleName)
	{
		return getAgentWithRole(communityName, groupName, roleName) != null;
		//		try {
		//		if(getAgentWithRole(communityName, groupName, roleName) != null)
		//		return true;
		//		} catch (OrganizationRequestException e) {
		//		return false;
		//		}
		//		return false;
	}
	/** @return <code>true</code> if the role exists (i.e. there is at least one agent with this role); <code> false </code>otherwise.*/
	public boolean isRole(final String groupName, final String roleName)
	{
		return isRole(Kernel.DEFAULT_COMMUNITY, groupName, roleName);
	}

	/** @deprecated As of MadKit 3.0. replaced by {@link #getExistingGroups(String)}*/
	public java.util.Vector<String> getGroups()
	{
		final String[] groups = currentKernel.getCurrentGroupsOf(getAddress(),Kernel.DEFAULT_COMMUNITY);
		final java.util.Vector<String> v = new java.util.Vector<String>();
		for(int i = 0;i < groups.length;i++)
			v.addElement(groups[i]);
		return v;
	}

	/** Determines if this group already exists in this community*/
	public boolean isGroup(final String communityName, final String groupName)
	{
		return currentKernel.isGroup(communityName, groupName);
	}
	/** Determines if this group already exists in the default community*/
	public boolean isGroup(final String groupName)
	{
		return currentKernel.isGroup(Kernel.DEFAULT_COMMUNITY, groupName);
	}

	/** Determines if this group already exists in this community
	@since MadKit 3.0
	 */
	public boolean isCommunity(final String communityName)
	{
		return currentKernel.isCommunity(communityName);
	}

	/** returns the available communities
	@since MadKit 3.0
	 */
	public String[] getAvailableCommunities()
	{
		return currentKernel.getCommunities();
	}

	/** returns true if the community is shared on the network
	@since MadKit 3.0
	 */
	public boolean connectedWithCommunity(final String communityName)
	{
		return currentKernel.connectedWithCommunity(communityName);
	}


	/** This method is still experimental. Use with caution ;)*/
	public void destroyGroup(final String communityName, final String groupName){
		currentKernel.destroyGroup(communityName,groupName);
	}


	///////////////////////////////////////////////////////// MESSAGING

	/** Is there any message left to be read ? */
	public boolean isMessageBoxEmpty(){	return messageBox.isEmpty();}

	/** Gets the number of messages in the message box */
	public int getMessageBoxSize()
	{
		return messageBox.size();//getSize();
	}

	/** Gets the first message in the queue.
	@return the first item in the message queue, or null if there is no message available
	 */
	public Message nextMessage()
	{
		if (! isMessageBoxEmpty())
			return messageBox.remove(0);
		return null;
	}

	//	return messageBox.nextMessage();}

	/**
	 * This method is invoked by the kernel when the agent receives a message.
	 * 
	 * @param m the message
	 */
	protected void receiveMessage(final Message m){    messageBox.add(m);}//}putMessage(m);}

	/**
	 * Send a message to another agent.
	 * 
	 * @param a AgentAddress of the receiver
	 * @param m Message to be sent
	 * 
	 * @throws MessageException if no recipient is found or the message is null
	 */
	public void sendMessage(final AgentAddress a, final Message m) throws MessageException
	{
		if(a != null){
			try{
				m.setReceiver(a);
			}
			catch (NullPointerException e) {
				throw new MessageException(getAddress()," Cannot send a null message",null);
			}
			m.setSender(getAddress());
			try {
				currentKernel.sendMessage(m);
			} 
			catch (MessageException e) {
				handleException(e);
			}
		}
		else
			try {
				throw new MessageException(getAddress()," Cannot send a message to a null recipient",null);
			} catch (MessageException e) {
				handleException(e);
				return;
			}
	}

	/**
	 * Send a message to an agent having a specific role in a group of this particular community. The corresponding AgentAddress is selected with a getAgentWithRole(..) method call.
	 * 
	 * @param communityName community in which the group is defined
	 * @param groupName group in which the role is defined
	 * @param roleName Role of the receiver
	 * @param m Message to be sent
	 * 
	 * @return <code>true</code>, if an agent has been found as receiver. Beware that this does not mean that the message will be successfully delivered.
	 * 
	 * @throws MessageException if the message is null
	 * 
	 * @see #getAgentWithRole(String, String, String)
	 * @since madkit 3.0
	 */
	public boolean sendMessage(final String communityName, final String groupName, final String roleName, final Message m) throws MessageException
	{
		if(! Kernel.interGroupMessage) //TODO à optimiser : une seule recherche sur le groupe
		{
			if(! currentKernel.isBelongingToGroup(getAddress(),communityName, groupName))
			{
				throw new MessageException(getAddress()," I am not allowed to send inter group messages !",null);
			}
		}
		final AgentAddress target = getAgentWithRole(communityName, groupName, roleName);
		if(target==null)
			try {
				throw new MessageException(getAddress()," Cannot send a message to a null recipient : no agent handles the role <"+communityName+","+groupName+","+roleName+">",null);
			} catch (MessageException e) {
				handleException(e);
				return false;
			}
			sendMessage(target, m);
			return true;
			//			try {
			//			final AgentAddress target = getAgentWithRole(communityName, groupName, roleName);
			//			if(target != null)
			//			m.setReceiver(target);
			//			else 
			//			throw new MessageException(getAddress()," Cannot send a message to a null recipient : no agent handles the role <"+communityName+","+groupName+","+roleName+">",null);
			//			} catch (NullPointerException e) {
			//			throw new MessageException(getAddress()," Cannot send a null message",(Madkit.debug)? e : null);
			//			}
			//			if (m.getReceiver() != null){
			//			m.setSender(getAddress());
			//			currentKernel.sendMessage(m);
			//			}
	}

	/**
	 * Send a message to an agent having a specific role in a group of the default community
	 * 
	 * @param groupName the group name
	 * @param roleName the role name
	 * @param m the message
	 * 
	 * @throws MessageException the message exception
	 * @see #sendMessage(String, String, String, Message)
	 */
	public void sendMessage(final String groupName, final String roleName, final Message m) throws MessageException
	{
		sendMessage(Kernel.DEFAULT_COMMUNITY, groupName, roleName, m);
	}

	/**
	 * Broadcast a message to every agent having a role in a group. You must assume that the message might not be cloned when sent to local agents
	 * 
	 * @param communityName the community name
	 * @param groupName the group name
	 * @param roleName the role name
	 * @param m the m
	 * 
	 * @throws MessageException the message exception
	 */
	public void broadcastMessage(final String communityName, final String groupName, final String roleName, final Message m)  throws MessageException
	{
		try {
			m.setSender(getAddress());
		} catch (NullPointerException e1) {
			throw new MessageException(getAddress()," Cannot broadcast a null message",null);
		}
		try {
			currentKernel.sendBroadcastMessage(communityName, groupName, roleName, m);
		} catch (MessageException e) {
			handleException(e);
		}
	}
	public void broadcastMessage(final String groupName, final String roleName, final Message m)  throws MessageException
	{
		broadcastMessage(Kernel.DEFAULT_COMMUNITY, groupName, roleName, m);
	}

	/**
	 * Basic handling of all messages that an agent may receive to ask it
	 * to perform various tasks. This method should be invoked after or before
	 * handling specific messages.
	 * 
	 * @param agent the agent
	 * @param name the name
	 * @param gui the gui
	 * 
	 * @return a boolean value which is true if the message has been handled
	 * and false otherwise.
	 */
	/* public boolean handleMessage(Message m){
    return false;
} */



	/////////////////////////////////////////////////////////////////////////////////////////////////
	///////////////////////////////////////////////////// AGENT LAUNCHING & KILLING
	/////////////////////////////////////////////////////////////////////////////////////////////

	/** This method is called in order to launch an agent from another agent.
     @param agent The new agent (already instantiated using a new instruction) .
     @param name The "usual" name
     @param gui Should we setup the agent GUI if possible ?
	 */
	public void launchAgent(final AbstractAgent agent, final String name, final boolean gui)
	{
		launchAgent(agent, name, gui ? new Point(-1,-1) : null, gui ? new Dimension(-1,-1) : null); // a fake point to have a gui
	}

	/**
	 * This method try to launch an agent with a GUI with the specified coordinates and dimension.
	 * 
	 * @param agent the agent
	 * @param name the name of the agent
	 * @param position the position on the screen
	 * @param dim the dimension of the GUI
	 */
	public void launchAgent(final AbstractAgent agent, String name, final Point position, final Dimension dim)
	{
		if (name == null)
			name = "unamed";
		try {
			currentKernel.launchAgent(agent, name, this, (position==null ? false : true), position, dim);
		} catch (LaunchAgentException e) {
			handleException(e);
		}
	}

	/**experimental*/
	public void restoreAgent(final AbstractAgent agent)
	{
		currentKernel.restoreAgent(agent,agent.getName(),this,agent.hasGUI());
	}

	/**
	 * Kills another agent (or self). This is only possible if the caller launched the target agent itself and still owns a reference to the potential victim
	 * 
	 * @param agent the agent to kill
	 */
	public void killAgent(final AbstractAgent agent) 
	{
		try{
			currentKernel.killAgent(agent, this);
		}
		catch(OrganizationRequestException e){
			handleException(e);
		}
	}
	///////////////////////////////////////////////////////////////////////////////////



	/////////////////////////////////////////////////////
	//////Accessors
	/////////////////////////////////////////////////////
	/**
	 * Gets the agent's own agent address.
	 * 
	 * @return the address
	 */
	//public AgentAddress getAddress() {return agentInformation.getAddress();}
	public AgentAddress getAddress() {return myAddress;}

	/**
	 * Gets the agent's own information.
	 * 
	 * @return the agent information
	 */
	//public AgentInformation getAgentInformation() {      return agentInformation;}

	/**
	 * Change the current agent patronymic name. This name does not need to be unique, and is provided only as a facility
	 * 
	 * @param theName the new name to set
	 */
	//	public void setName(final String theName){    agentInformation.setName(theName);}
	public void setName(final String theName){    myAddress.setName(theName);}

	/**
	 * Gets the current agent patronymic name.
	 * 
	 * @return the name of the agent
	 */
	//public String getName(){    return agentInformation.getName();}
	public String getName(){    return myAddress.getName();}

	/////////////////////////////////////////////   INTERNAL PACKAGE METHODS
	/**
	 * Sets the current kernel.
	 * 
	 * @param theKernel the new current kernel
	 */
	final void setCurrentKernel(final Kernel theKernel){	currentKernel = theKernel;}

	/**
	 * internal package method for some special agents.
	 * 
	 * @return the current kernel
	 */
	final Kernel getCurrentKernel(){	return currentKernel;}

	/**
	 * Internal agentInformation direct access method.
	 * 
	 * @param agentInfo the agent info
	 */
	//final void setAgentInformation(final AgentInformation agentInfo){agentInformation = agentInfo;}

	/** 
	 * Returns the kernelAddress of the agency where this agent is situated... 
	 * Generally this value is equal to getAddress().getKernel() but when an agent migrate, this value may change.
	 * This method is used to get the real value of the current kernel address.
	 * @return the current kernelAddress
	 */
	public KernelAddress getCurrentKernelAddress(){
		return Kernel.getAddress();
	}


	///////////////////////// GRAPHICS Agent Graphical Componential Interface
	/** Check if a bean is running in GUI mode. Usually verified by the Kernel or the host application to setup or not a default graphical interface
	@return true if a GUI has been instantiated
	 */
	public boolean hasGUI(){    return (currentBean!=null);}

	/** This method is called by the specific external graphic system (as the G-Box)
to ask the agent to prepare a graphical interface. The agent developer should
use a setGUIObject(...) within this method, as well as other necessary
initializations. If the developer does not overload this method, a vanilla text
output might be used as the default interface. */
	public void initGUI()
	{
		if (currentKernel != null)
			setGUIObject(currentKernel.gui.getDefaultGUIObject(this));
	}

	/** @deprecated As of MadKit 2.0. replaced by {@link #setGUIObject(Object)}*/
	public void setBean(final Object theBean)
	{
		currentBean = theBean;
		//		bean_mode = true;
	}

	/**@deprecated As of MadKit 2.0. replaced by {@link #getGUIObject()}*/
	public Object getBean(){return currentBean;}

	/** This method set the bean that will be used to represent the agent in a graphical environment, the agent is also registered as running in GUI mode
	@param theBean an allocated graphic component
	@since MadKit 2.0
	 */
	public void setGUIObject(final Object theBean)
	{
		currentBean = theBean;
		//bean_mode = true;
	}

	/**
	 * Gets the graphic component representing the agent. Usually called by the host application
	 * 
	 * @return the GUI object
	 * 
	 * @since MadKit 2.0
	 */
	public Object getGUIObject()
	{
		return currentBean;
	}

	/**
	 * Print out debug information only if the debug flag is on.
	 * 
	 * @param message the message
	 */
	public void debug(final String message)
	{
		if(debugFlag)
			try
		{
				if (ostream != null)
					ostream.write("*Debug* "+message);
				else
					System.err.println("["+getAddress()+"] : *Debug* "+message);
		}
		catch (final IOException e)
		{
			currentKernel.displayln("IOException println: "+e.toString());
			e.printStackTrace();
		}
	}

	/**
	 * Sets the debug flag.
	 * 
	 * @param b the b
	 */
	public void setDebug(final boolean b){ debugFlag = b;}
	/**@return  true if debug is on*/
	public boolean getDebug(){return debugFlag;}

	/**
	 * Prints text information in an environment-independent way (GUI, console, ...)
	 * 
	 * @param message the string
	 */
	public void println(final String message)
	{
		print(message+"\n");
	}

	/**
	 * Prints text information in a environment-independent way (GUI, console, ...)
	 * 
	 * @param message the string
	 */
	public void print(final String message)
	{
		try
		{
			if (ostream != null)
				ostream.write(message);
			else
				System.err.println("["+getName()+"] : "+message);
		}
		catch (final IOException e)
		{
			currentKernel.displayln("IOException println: "+e.toString());
			e.printStackTrace();
		}
	}


	/** @deprecated As of MadKit 2.0. replaced by {@link #setOutputWriter(Writer)}*/
	public void setOutput(final Writer o){setOutputWriter(o);}
	/** Reassigns the "standard" agent text output stream (used by method println).*/
	public void setOutputWriter(final Writer o){ostream = o;}

	/** hide the graphical component that represents the Agent in some higher interface
	@since MadKit 3.0
	 */
	public void disposeMyGUI()
	{
		setOutputWriter(null);
		//bean_mode=false;
		currentKernel.disposeGUIOf(this);
	}
	
	public void hideMyGUI()
	{
		currentKernel.disposeGUIOf(this);
	}

	/** try to restore the graphical component that represents the Agent in some higher interface
	@since MadKit 3.0
	 */
	public void redisplayMyGUI()
	{
		currentKernel.redisplayGUIOf(this);
	}

	public void windowClosing(final AWTEvent we){
		if(currentKernel != null) //already dead another way
			killAgent(this);
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	final public int hashCode() {
		return _hashCode;
	}

	@Override
	public String toString() {
		return getAddress().toString();
	}


	final private int handleException(MadkitException e) {
		if(Madkit.outputOnRequestErrors){
			println(e.getClass().getSimpleName()+e.getMessage());
			if(getDebug())
				e.printStackTrace();
		}
		return e.getCode();
	}

	/**
	 * @param myAddress the myAddress to set
	 */
	final void setMyAddress(AgentAddress myAddress) {
		this.myAddress = myAddress;
	}

	/**
	 * @return the agent thread
	 */
	Thread getAgentThread()
	{
		return null;
	}


}
