/**
 * ***************************************************************
 * ContentManager.java
 * 
 * GNU Lesser General Public License
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation,
 * version 2.1 of the License.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 * **************************************************************
 */
package madkit.fipa.content;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;

import madkit.fipa.CaseInsensitiveString;
import madkit.fipa.abs.AbsConcept;
import madkit.fipa.abs.AbsContentElement;
import madkit.fipa.abs.AbsContentElementList;
import madkit.fipa.abs.AbsObject;
import madkit.messages.ACLMessage;

import org.omg.IOP.Codec;

/**
 *
 * Cette classe fournit toutes les m�thodes pour g�rer les langages de contenu 
 * et les ontologies connus par un agent donn�. 
 * Il est possible d'enregistrer plusieurs ontologies et langages.
 * Les m�thodes fillContent() et ExtractContent() permettent d'extraire et de remplir le champ 
 * contenu du message ACL selon le langage choisi ainsi que l'ontologie.
 * Chaque agent acc�de � son content Manager par la m�thode getContentManager()
 * de la classe FIPAAgent.
 * Il est possible de d�savtiver la validation de l'objet par rapport au sch�ma 
 * afin d'augmenter le performance.
 * 
 * This class provides all methods to manage the content languages 
 * and ontologies "known" by a given agent and to fill and extract the 
 * content of an ACL message according to a given content language and 
 * ontology.
 * Each agent has a <code>ContentManager</code> object accessible through
 * the <code>getContentManager()</code> method of the <code>Agent</code>
 * class.
 * 
 * @author Federico Bergenti
 * @author Govanni Caire - TILAB
 */
public class ContentManager implements Serializable {
    transient private Map languages = new HashMap();
    transient private Map ontologies = new HashMap();
    private boolean validationMode = true;
    
    //#MIDP_EXCLUDE_BEGIN
    private void readObject(java.io.ObjectInputStream oin) throws java.io.IOException, ClassNotFoundException {
	oin.defaultReadObject();
	languages = new HashMap();
	ontologies = new HashMap();
    }
    //#MIDP_EXCLUDE_END

    /**
     * Registers a <code>Codec</code> for a given content language 
     * with its default name (i.e.
     * the name returned by its <code>getName()</code> method.
     * Since this operation is performed the agent that owns this 
     * <code>ContentManager</code> is able to "speak" the language
     * corresponding to the registered <code>Codec</code>.
     * @param c the <code>Codec</code> to be registered.
     */
    public void registerLanguage(Codec c) {
    	if (c == null) {
    		throw new IllegalArgumentException("Null codec registered");
    	}
    	registerLanguage(c, c.getName());
    }

    /**
     * Registers a <code>Codec</code> for a given content language 
     * with a given name.
     * @param c the <code>Codec</code> to be registered.
     * @param name the name associated to the registered codec.
     */
    public void registerLanguage(Codec c, String name) {
    	if (c == null) {
    		throw new IllegalArgumentException("Null codec registered");
    	}
      languages.put(new CaseInsensitiveString(name), c);
    }

    /**
     * Registers an <code>Ontology</code> with its default name (i.e.
     * the name returned by its <code>getName()</code> method.
     * Since this operation is performed the agent that owns this 
     * <code>ContentManager</code> "knows" the registered 
     * <code>Ontology</code>.
     * @param o the <code>Ontology</code> to be registered.
     */
    public void registerOntology(Ontology o) {
    	if (o == null) {
    		throw new IllegalArgumentException("Null ontology registered");
    	}
      registerOntology(o, o.getName());
    }

    /**
     * Registers an <code>Ontology</code> with a given name.
     * @param o the <code>Ontology</code> to be registered.
     * @param name the name associated to the registered Ontology.
     */
    public void registerOntology(Ontology o, String name) {
    	if (o == null) {
    		throw new IllegalArgumentException("Null ontology registered");
    	}
      ontologies.put(new CaseInsensitiveString(name), o);
    }

    /**
     * Retrieves a previously registered <code>Codec</code> 
     * giving its <code>name</code>.
     * @param name the name associated to the <code>Codec</code> 
     * to be retrieved.
     * @return the <code>Codec</code> associated to 
     * <code>name</code> or <code>null</code> if no Codec was registered 
     * with the given name.
     */
    public Codec lookupLanguage(String name) {
        return (Codec) languages.get(new CaseInsensitiveString(name));
    }

    /**
     * Retrieves a previously registered <code>Ontology</code> 
     * giving its <code>name</code>.
     * @param name the name associated to the <code>Ontology</code> 
     * to be retrieved.
     * @return the <code>Ontology</code> associated to 
     * <code>name</code> or <code>null</code> if no Ontology was registered 
     * with the given name.
     */
    public Ontology lookupOntology(String name) {
        return (Ontology) ontologies.get(new CaseInsensitiveString(name));
    }

    /**
     * Fills the <code>:content</code> slot of an 
     * <code>ACLMessage msg</code> using the content language
     * and ontology indicated in the <code>:language</code> and 
     * <code>:ontology</code> fields of <code>msg</code>.
     * @param msg the message whose content has to be filled.
     * @param content the content of the message represented as an 
     * <code>AbsContentElement</code>.
     * @throws CodecException if <code>content</code> is not compliant
     * to the content language used for this operation.
     * @throws OntologyException if <code>content</code> is not compliant
     * to the ontology used for this operation.
     */
    public void fillContent(ACLMessage msg, AbsContentElement content) 
            throws CodecException, OntologyException {
        Codec    codec = lookupLanguage(msg.getLanguage());
    		if (codec == null) {
    			throw new CodecException("Unknown language "+msg.getLanguage());
    		}
    		Ontology o = lookupOntology(msg.getOntology());
    		if (o == null) {
    			throw new OntologyException("Unknown ontology "+msg.getOntology());
    		}
        Ontology onto  = getMergedOntology(codec, o);
        
        validate(content, onto);

        encode(msg, content, codec, onto);
    } 

    /**
     * Fills the <code>:content</code> slot of an 
     * <code>ACLMessage msg</code> using the content language
     * and ontology indicated in the <code>:language</code> and 
     * <code>:ontology</code> fields of <code>msg</code>.
     * @param msg the message whose content has to be filled.
     * @param content the content of the message represented as a 
     * <code>ContentElement</code>.
     * @throws CodecException if <code>content</code> is not compliant
     * to the content language used for this operation.
     * @throws OntologyException if <code>content</code> is not compliant
     * to the ontology used for this operation.
     Remplit le champs content d'un ACLmessage � partir des objets java,
     de l'ontologie et du codec
     */
    public void fillContent(ACLMessage msg, ContentElement content) 
            throws CodecException, OntologyException {
        Codec    codec = lookupLanguage(msg.getLanguage());
       // System.out.println(msg.getLanguage());
    		if (codec == null) {
    			throw new CodecException("Unknown language "+msg.getLanguage());
    		}
    		Ontology o = lookupOntology(msg.getOntology());
    		if (o == null) {
    			throw new OntologyException("Unknown ontology "+msg.getOntology());
    		}
        Ontology onto  = getMergedOntology(codec, o);
        
        //On transforme les objets java en elements abstrait
        // 1ere partie du pipeline de conversion
        AbsContentElement abs = (AbsContentElement) onto.fromObject(content);
       	//abs.setTypeName(abs.setTypeName().trim());
       	System.out.println("classe test "+onto.toObject(abs)+" "+onto.getName()+" "+abs.getTypeName());
        System.out.println(abs+" / "+abs.getCount()+" /"+abs.getTypeName()+"/");
        
        AbsConcept absp = (AbsConcept)abs.getAbsObject(abs.getNames()[0]);
		AbsObject abso = absp.getAbsObject(absp.getNames()[0]);
        AbsObject absl = abso.getAbsObject(abso.getNames()[0]);
        
        AbsObject abso2 = absp.getAbsObject(absp.getNames()[1]);
       
        System.out.println(abs.getClass() +" "+abs.getTypeName()+ "\n " + absp.getClass()+" "+absp.getTypeName()+" "+absp.getNames()[0]+"\n "+abso.getClass()+" "+abso.getTypeName()+"\n "+absl.getClass()+" "+absl.getTypeName()+" "+absl.getCount());
        System.out.println(abso2.getClass()+" "+abso2.getTypeName()+abso2.getCount());
        //System.out.println( ((AbsAgentAction)abso).getString("name"));
        // Ensuite une validation optionnelle entre les schemas et les elmt abstraits
        validate(abs, onto);
        // enfin les elemnt abstraits sont transform�s en String au format SL
        encode(msg, abs, codec, onto);
    } 

    /**
     * Translates the <code>:content</code> slot of an 
     * <code>ACLMessage msg</code> into an <code>AbsContentElement</code>
     * using the content language and ontology indicated in the 
     * <code>:language</code> and <code>:ontology</code> fields of <code>msg</code>.
     * @param msg the message whose content has to be extracted.
     * @return the content of the message represented as an 
     * <code>AbsContentElement</code>.
     * @throws CodecException if the content of the message is not compliant
     * to the content language used for this operation.
     * @throws OntologyException if the content of the message is not compliant
     * to the ontology used for this operation.
     */
    public AbsContentElement extractAbsContent(ACLMessage msg) 
            throws CodecException, OntologyException {
        Codec    codec = lookupLanguage(msg.getLanguage());
    		if (codec == null) {
    			throw new CodecException("Unknown language "+msg.getLanguage());
    		}
    		Ontology o = lookupOntology(msg.getOntology());
    		if (o == null) {
    			throw new OntologyException("Unknown ontology "+msg.getOntology());
    		}
        Ontology onto  = getMergedOntology(codec, o);
        
        AbsContentElement content = decode(msg, codec, onto);
        
        validate(content, onto);
        
        return content;
    }

    /**
     * Translates the <code>:content</code> slot of an 
     * <code>ACLMessage msg</code> into a <code>ContentElement</code>
     * using the content language and ontology indicated in the 
     * <code>:language</code> and <code>:ontology</code> fields of <code>msg</code>.
     * @param msg the message whose content has to be extracted.
     * @return the content of the message represented as a 
     * <code>ContentElement</code>.
     * @throws CodecException if the content of the message is not compliant
     * to the content language used for this operation.
     * @throws OntologyException if the content of the message is not compliant
     * to the ontology used for this operation.
     */
    public ContentElement extractContent(ACLMessage msg) 
            throws CodecException, UngroundedException, OntologyException {
        Codec    codec = lookupLanguage(msg.getLanguage());
    		if (codec == null) {
    			throw new CodecException("Unknown language "+msg.getLanguage());
    		}
    		Ontology o = lookupOntology(msg.getOntology());
    		if (o == null) {
    			throw new OntologyException("Unknown ontology "+msg.getOntology());
    		}
        Ontology onto  = getMergedOntology(codec, o);
        
        AbsContentElement content = decode(msg, codec, onto);
        content = (AbsContentElement)((AbsContentElementList)content).get(0);
        System.out.println(onto.getName());
        validate(content, onto);
         System.out.println("retour  /" + content.getTypeName()+"/");
       
		return (ContentElement) onto.toObject(content);
    } 

		/** 
		   Set the validation mode i.e. whether contents that are managed
		   by this content manager should be validated during 
		   message content filling/extraction.
		   Default value is <code>true</code>
		   @param mode the new validation mode 
		 */
		public void setValidationMode(boolean mode) {
			validationMode = mode;
		}
		
		/** 
		   Return the currently set validation mode i.e. whether 
		   contents that are managed by this content manager should 
		   be validated during message content filling/extraction.
		   Default value is <code>true</code>
		   @return the currently set validation mode 
		 */
		public boolean getValidationMode() {
			return validationMode;
		}
		
		/** 
		 */
		public Ontology getOntology(ACLMessage msg) {
      Codec    codec = lookupLanguage(msg.getLanguage());
      return getMergedOntology(codec, lookupOntology(msg.getOntology()));
		}
		
		/**
		 * Merge the reference ontology with the inner ontology of the 
		 * content language
		 */
    private Ontology getMergedOntology(Codec c, Ontology o) {
				Ontology ontology = null;
				Ontology langOnto = c.getInnerOntology();
				if (langOnto == null) {
					ontology = o;
				}
				else {
					ontology = new Ontology(null, new Ontology[]{o, langOnto}, null);
				}
				return ontology;
    }
    
		private void validate(AbsContentElement content, Ontology onto) throws OntologyException { 
    	if (validationMode) {
        // Validate the content against the ontology
    		ObjectSchema schema = onto.getSchema(content.getTypeName());
    		System.out.println("validate "+onto.getName()+" "+content.getTypeName());
   			if (schema == null) {
  				throw new OntologyException("No schema found for type "+content.getTypeName());
  			}
    		schema.validate(content, onto);
      }
		}

		private void encode(ACLMessage msg, AbsContentElement content, Codec codec, Ontology onto) throws CodecException, OntologyException { 
      		if (codec instanceof ByteArrayCodec)
				msg.setByteSequenceContent(((ByteArrayCodec) codec).encode(onto, content));
			else if (codec instanceof StringCodec)
				msg.setContent(((StringCodec) codec).encode(onto, content));
			else
				throw new CodecException("UnsupportedTypeOfCodec");
		}

		private AbsContentElement decode(ACLMessage msg, Codec codec, Ontology onto) throws CodecException, OntologyException { 
      		if (codec instanceof ByteArrayCodec)
				return ((ByteArrayCodec) codec).decode(onto, msg.getByteSequenceContent());
			else if (codec instanceof StringCodec)
				return ((StringCodec) codec).decode(onto, msg.getContent());
			else
				throw new CodecException("UnsupportedTypeOfCodec");
		}
		
}

